-- Create Oracle New User and Connect
--------------------------------------
-- CREATE USER apimuser IDENTIFIED BY wso2;
-- GRANT CONNECT,RESOURCE to apimuser WITH ADMIN OPTION;
-- CONNECT apimuser/wso2
--

-- Start of IDN Tables --
CREATE TABLE IDN_BASE_TABLE (
            PRODUCT_NAME VARCHAR2 (20),
            PRIMARY KEY (PRODUCT_NAME))
/
CREATE TABLE IDN_OAUTH_CONSUMER_APPS (
            ID INTEGER,
            CONSUMER_KEY VARCHAR2 (255),
            CONSUMER_SECRET VARCHAR2 (2048),
            USERNAME VARCHAR2 (255),
            TENANT_ID INTEGER DEFAULT 0,
            USER_DOMAIN VARCHAR(50),
            APP_NAME VARCHAR2 (255),
            OAUTH_VERSION VARCHAR2 (128),
            CALLBACK_URL VARCHAR2 (2048),
            GRANT_TYPES VARCHAR (1024),
            PKCE_MANDATORY CHAR(1) DEFAULT '0',
            PKCE_SUPPORT_PLAIN CHAR(1) DEFAULT '0',
            APP_STATE VARCHAR (25) DEFAULT 'ACTIVE',
            USER_ACCESS_TOKEN_EXPIRE_TIME NUMBER(19) DEFAULT 3600,
            APP_ACCESS_TOKEN_EXPIRE_TIME NUMBER(19) DEFAULT 3600,
            REFRESH_TOKEN_EXPIRE_TIME NUMBER(19) DEFAULT 84600,
            ID_TOKEN_EXPIRE_TIME NUMBER(19) DEFAULT 3600,
            CONSTRAINT CONSUMER_KEY_CONSTRAINT UNIQUE (CONSUMER_KEY),
            PRIMARY KEY (ID))
/
CREATE TABLE IDN_OAUTH2_SCOPE_VALIDATORS (
	APP_ID INTEGER NOT NULL,
	SCOPE_VALIDATOR VARCHAR (128) NOT NULL,
	PRIMARY KEY (APP_ID,SCOPE_VALIDATOR),
	FOREIGN KEY (APP_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID)
)
/
CREATE TABLE IDN_OAUTH1A_REQUEST_TOKEN (
            REQUEST_TOKEN VARCHAR2 (512),
            REQUEST_TOKEN_SECRET VARCHAR2 (512),
            CONSUMER_KEY_ID INTEGER,
            CALLBACK_URL VARCHAR2 (2048),
            SCOPE VARCHAR2(2048),
            AUTHORIZED VARCHAR2 (128),
            OAUTH_VERIFIER VARCHAR2 (512),
            AUTHZ_USER VARCHAR2 (512),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (REQUEST_TOKEN),
            FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID))
/
CREATE TABLE IDN_OAUTH1A_ACCESS_TOKEN (
            ACCESS_TOKEN VARCHAR2 (512),
            ACCESS_TOKEN_SECRET VARCHAR2 (512),
            CONSUMER_KEY_ID INTEGER,
            SCOPE VARCHAR2(2048),
            AUTHZ_USER VARCHAR2 (512),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (ACCESS_TOKEN),
            FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID))
/
CREATE TABLE IDN_OAUTH2_ACCESS_TOKEN (
            TOKEN_ID VARCHAR (255),
            ACCESS_TOKEN VARCHAR2 (2048),
            REFRESH_TOKEN VARCHAR2 (2048),
            CONSUMER_KEY_ID INTEGER,
            AUTHZ_USER VARCHAR2 (100),
            TENANT_ID INTEGER,
            USER_DOMAIN VARCHAR2 (50),
            USER_TYPE VARCHAR (25),
            GRANT_TYPE VARCHAR (50),
            TIME_CREATED TIMESTAMP,
            REFRESH_TOKEN_TIME_CREATED TIMESTAMP,
            VALIDITY_PERIOD NUMBER(19),
            REFRESH_TOKEN_VALIDITY_PERIOD NUMBER(19),
            TOKEN_SCOPE_HASH VARCHAR2 (32),
            TOKEN_STATE VARCHAR2 (25) DEFAULT 'ACTIVE',
            TOKEN_STATE_ID VARCHAR (128) DEFAULT 'NONE',
            SUBJECT_IDENTIFIER VARCHAR(255),
            ACCESS_TOKEN_HASH VARCHAR2 (512),
            REFRESH_TOKEN_HASH VARCHAR2 (512),
            IDP_ID INTEGER,
            TOKEN_BINDING_REF VARCHAR2 (32) DEFAULT 'NONE',
            DCID VARCHAR2 (50) DEFAULT 'DC1',
            PRIMARY KEY (TOKEN_ID),
            FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID),
            CONSTRAINT CON_APP_KEY UNIQUE (CONSUMER_KEY_ID,AUTHZ_USER,TENANT_ID,USER_DOMAIN,USER_TYPE,TOKEN_SCOPE_HASH,
                                           TOKEN_STATE,TOKEN_STATE_ID,IDP_ID,TOKEN_BINDING_REF,DCID))
/
CREATE TABLE IDN_OAUTH2_TOKEN_BINDING (
            TOKEN_ID VARCHAR2 (255),
            TOKEN_BINDING_TYPE VARCHAR2 (32),
            TOKEN_BINDING_REF VARCHAR2 (32),
            TOKEN_BINDING_VALUE VARCHAR2 (1024),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (TOKEN_ID),
            FOREIGN KEY (TOKEN_ID) REFERENCES IDN_OAUTH2_ACCESS_TOKEN(TOKEN_ID))
/
CREATE TABLE IDN_OAUTH2_ACCESS_TOKEN_AUDIT (
            TOKEN_ID VARCHAR (255),
            ACCESS_TOKEN VARCHAR2(2048),
            REFRESH_TOKEN VARCHAR2(2048),
            CONSUMER_KEY_ID INTEGER,
            AUTHZ_USER VARCHAR2 (100),
            TENANT_ID INTEGER,
            USER_DOMAIN VARCHAR2(50),
            USER_TYPE VARCHAR (25),
            GRANT_TYPE VARCHAR (50),
            TIME_CREATED TIMESTAMP NULL,
            REFRESH_TOKEN_TIME_CREATED TIMESTAMP NULL,
            VALIDITY_PERIOD NUMBER(19),
            REFRESH_TOKEN_VALIDITY_PERIOD NUMBER(19),
            TOKEN_SCOPE_HASH VARCHAR2(32),
            TOKEN_STATE VARCHAR2(25),
            TOKEN_STATE_ID VARCHAR (128) ,
            SUBJECT_IDENTIFIER VARCHAR(255),
            ACCESS_TOKEN_HASH VARCHAR2(512),
            REFRESH_TOKEN_HASH VARCHAR2(512),
            INVALIDATED_TIME TIMESTAMP NULL,
            IDP_ID INTEGER)
/
CREATE TABLE IDN_OAUTH2_AUTHORIZATION_CODE (
            CODE_ID VARCHAR (255),
            AUTHORIZATION_CODE VARCHAR2 (2048),
            CONSUMER_KEY_ID INTEGER,
	        CALLBACK_URL VARCHAR2 (2048),
            SCOPE VARCHAR2(2048),
            AUTHZ_USER VARCHAR2 (100),
            TENANT_ID INTEGER,
            USER_DOMAIN VARCHAR2 (50),
            TIME_CREATED TIMESTAMP,
            VALIDITY_PERIOD NUMBER(19),
            STATE VARCHAR (25) DEFAULT 'ACTIVE',
            TOKEN_ID VARCHAR(255),
            SUBJECT_IDENTIFIER VARCHAR(255),
            PKCE_CODE_CHALLENGE VARCHAR (255),
            PKCE_CODE_CHALLENGE_METHOD VARCHAR(128),
            AUTHORIZATION_CODE_HASH VARCHAR2 (512),
            IDP_ID INTEGER,
            PRIMARY KEY (CODE_ID),
            FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID))

/
CREATE TABLE IDN_OAUTH2_AUTHZ_CODE_SCOPE(
            CODE_ID   VARCHAR2(255),
            SCOPE     VARCHAR2(60),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (CODE_ID, SCOPE),
            FOREIGN KEY (CODE_ID) REFERENCES IDN_OAUTH2_AUTHORIZATION_CODE (CODE_ID))
/

CREATE TABLE IDN_OAUTH2_DEVICE_FLOW (
            CODE_ID VARCHAR2(255),
            DEVICE_CODE VARCHAR2(255),
            USER_CODE VARCHAR2(25),
            CONSUMER_KEY_ID INTEGER,
            LAST_POLL_TIME TIMESTAMP NOT NULL,
            EXPIRY_TIME TIMESTAMP NOT NULL,
            TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
            POLL_TIME NUMBER(19),
            STATUS VARCHAR2 (25) DEFAULT 'PENDING',
            AUTHZ_USER VARCHAR2 (100),
            TENANT_ID INTEGER,
            USER_DOMAIN VARCHAR2(50),
            IDP_ID INTEGER,
            PRIMARY KEY (DEVICE_CODE),
            UNIQUE (CODE_ID),
            FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID))
/

CREATE TABLE IDN_OAUTH2_DEVICE_FLOW_SCOPES (
            ID INTEGER NOT NULL,
            SCOPE_ID VARCHAR2(255),
            SCOPE VARCHAR2(255),
            PRIMARY KEY (ID),
            FOREIGN KEY (SCOPE_ID) REFERENCES IDN_OAUTH2_DEVICE_FLOW(CODE_ID) )
/
CREATE TABLE IDN_OAUTH2_ACCESS_TOKEN_SCOPE (
            TOKEN_ID VARCHAR2 (255),
            TOKEN_SCOPE VARCHAR2 (60),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (TOKEN_ID, TOKEN_SCOPE),
            FOREIGN KEY (TOKEN_ID) REFERENCES IDN_OAUTH2_ACCESS_TOKEN(TOKEN_ID))
/
CREATE TABLE IDN_OAUTH2_SCOPE (
            SCOPE_ID INTEGER NOT NULL,
            NAME VARCHAR2(255) NOT NULL,
            DISPLAY_NAME VARCHAR2(255) NOT NULL,
            DESCRIPTION VARCHAR2(512),
            TENANT_ID INTEGER DEFAULT -1 NOT NULL,
            SCOPE_TYPE VARCHAR2(255) NOT NULL,
            PRIMARY KEY (SCOPE_ID))
/
CREATE TABLE IDN_OAUTH2_SCOPE_BINDING (
            SCOPE_ID INTEGER NOT NULL,
            SCOPE_BINDING VARCHAR2(255) NOT NULL,
            BINDING_TYPE VARCHAR2(255) NOT NULL,
            FOREIGN KEY (SCOPE_ID) REFERENCES IDN_OAUTH2_SCOPE(SCOPE_ID))
/
CREATE TABLE IDN_OAUTH2_RESOURCE_SCOPE (
            RESOURCE_PATH VARCHAR2 (255) NOT NULL,
            SCOPE_ID INTEGER NOT NULL,
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (RESOURCE_PATH),
            FOREIGN KEY (SCOPE_ID) REFERENCES IDN_OAUTH2_SCOPE (SCOPE_ID))
/
CREATE TABLE IDN_SCIM_GROUP (
			ID INTEGER,
			TENANT_ID INTEGER NOT NULL,
			ROLE_NAME VARCHAR2(255) NOT NULL,
            ATTR_NAME VARCHAR2(1024) NOT NULL,
			ATTR_VALUE VARCHAR2(1024),
            PRIMARY KEY (ID))

/
CREATE TABLE IDN_OPENID_REMEMBER_ME (
            USER_NAME VARCHAR(255) NOT NULL,
            TENANT_ID INTEGER DEFAULT 0,
            COOKIE_VALUE VARCHAR(1024),
            CREATED_TIME TIMESTAMP,
            PRIMARY KEY (USER_NAME, TENANT_ID))
/
CREATE TABLE IDN_OPENID_USER_RPS (
			USER_NAME VARCHAR(255) NOT NULL,
			TENANT_ID INTEGER DEFAULT 0,
			RP_URL VARCHAR(255) NOT NULL,
			TRUSTED_ALWAYS VARCHAR(128) DEFAULT 'FALSE',
			LAST_VISIT DATE NOT NULL,
			VISIT_COUNT INTEGER DEFAULT 0,
			DEFAULT_PROFILE_NAME VARCHAR(255) DEFAULT 'DEFAULT',
			PRIMARY KEY (USER_NAME, TENANT_ID, RP_URL))
/
CREATE TABLE IDN_OPENID_ASSOCIATIONS (
			HANDLE VARCHAR(255) NOT NULL,
			ASSOC_TYPE VARCHAR(255) NOT NULL,
			EXPIRE_IN TIMESTAMP NOT NULL,
			MAC_KEY VARCHAR(255) NOT NULL,
			ASSOC_STORE VARCHAR(128) DEFAULT 'SHARED',
			TENANT_ID INTEGER DEFAULT -1,
			PRIMARY KEY (HANDLE))
/
CREATE TABLE IDN_STS_STORE (
            ID INTEGER,
            TOKEN_ID VARCHAR(255) NOT NULL,
            TOKEN_CONTENT BLOB NOT NULL,
            CREATE_DATE TIMESTAMP NOT NULL,
            EXPIRE_DATE TIMESTAMP NOT NULL,
            STATE INTEGER DEFAULT 0,
            PRIMARY KEY (ID))
/
CREATE TABLE IDN_IDENTITY_USER_DATA (
            TENANT_ID INTEGER DEFAULT -1234,
            USER_NAME VARCHAR(255) NOT NULL,
            DATA_KEY VARCHAR(255) NOT NULL,
            DATA_VALUE VARCHAR(2048),
            PRIMARY KEY (TENANT_ID, USER_NAME, DATA_KEY))
/
CREATE TABLE IDN_IDENTITY_META_DATA (
            USER_NAME VARCHAR(255) NOT NULL,
            TENANT_ID INTEGER DEFAULT -1234,
            METADATA_TYPE VARCHAR(255) NOT NULL,
            METADATA VARCHAR(255) NOT NULL,
            VALID VARCHAR(255) NOT NULL,
            PRIMARY KEY (TENANT_ID, USER_NAME, METADATA_TYPE,METADATA))
/
CREATE TABLE IDN_THRIFT_SESSION (
            SESSION_ID VARCHAR2(255) NOT NULL,
            USER_NAME VARCHAR2(255) NOT NULL,
            CREATED_TIME VARCHAR2(255) NOT NULL,
            LAST_MODIFIED_TIME VARCHAR2(255) NOT NULL,
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (SESSION_ID)
)
/
CREATE TABLE IDN_AUTH_SESSION_STORE (
            SESSION_ID VARCHAR (100) NOT NULL,
            SESSION_TYPE VARCHAR(100) NOT NULL,
            OPERATION VARCHAR(10) NOT NULL,
            SESSION_OBJECT BLOB,
            TIME_CREATED NUMBER(19),
            TENANT_ID INTEGER DEFAULT -1,
            EXPIRY_TIME NUMBER(19),
            PRIMARY KEY (SESSION_ID, SESSION_TYPE, TIME_CREATED, OPERATION)
)

/
CREATE TABLE IDN_AUTH_TEMP_SESSION_STORE (
            SESSION_ID VARCHAR (100) NOT NULL,
            SESSION_TYPE VARCHAR(100) NOT NULL,
            OPERATION VARCHAR(10) NOT NULL,
            SESSION_OBJECT BLOB,
            TIME_CREATED NUMBER(19),
            TENANT_ID INTEGER DEFAULT -1,
            EXPIRY_TIME NUMBER(19),
            PRIMARY KEY (SESSION_ID, SESSION_TYPE, TIME_CREATED, OPERATION)
)

/

CREATE TABLE IDN_AUTH_USER (
            USER_ID VARCHAR(255) NOT NULL,
            USER_NAME VARCHAR(255) NOT NULL,
            TENANT_ID INTEGER NOT NULL,
            DOMAIN_NAME VARCHAR(255) NOT NULL,
            IDP_ID INTEGER NOT NULL,
            PRIMARY KEY (USER_ID),
            CONSTRAINT USER_STORE_CONSTRAINT UNIQUE (USER_NAME, TENANT_ID, DOMAIN_NAME, IDP_ID)
)
/

CREATE TABLE IDN_AUTH_USER_SESSION_MAPPING (
            USER_ID VARCHAR(255) NOT NULL,
            SESSION_ID VARCHAR(255) NOT NULL,
            CONSTRAINT USER_SESSION_STORE_CONSTRAINT UNIQUE (USER_ID, SESSION_ID)
)
/
CREATE TABLE IDN_AUTH_SESSION_APP_INFO (
            SESSION_ID VARCHAR (100) NOT NULL,
            SUBJECT VARCHAR (100) NOT NULL,
            APP_ID INTEGER NOT NULL,
            INBOUND_AUTH_TYPE VARCHAR (255) NOT NULL,
            PRIMARY KEY (SESSION_ID, SUBJECT, APP_ID, INBOUND_AUTH_TYPE)
)
/
CREATE TABLE IDN_AUTH_SESSION_META_DATA (
            SESSION_ID VARCHAR (100) NOT NULL,
            PROPERTY_TYPE VARCHAR (100) NOT NULL,
            VALUE VARCHAR (255) NOT NULL,
            PRIMARY KEY (SESSION_ID, PROPERTY_TYPE, VALUE)
)
/
CREATE TABLE SP_APP (
            ID INTEGER,
            TENANT_ID INTEGER NOT NULL,
            APP_NAME VARCHAR (255) NOT NULL ,
            USER_STORE VARCHAR (255) NOT NULL,
            USERNAME VARCHAR (255) NOT NULL ,
            DESCRIPTION VARCHAR (1024),
            ROLE_CLAIM VARCHAR (512),
            AUTH_TYPE VARCHAR (255) NOT NULL,
            PROVISIONING_USERSTORE_DOMAIN VARCHAR (512),
            IS_LOCAL_CLAIM_DIALECT CHAR(1) DEFAULT '1',
            IS_SEND_LOCAL_SUBJECT_ID CHAR(1) DEFAULT '0',
            IS_SEND_AUTH_LIST_OF_IDPS CHAR(1) DEFAULT '0',
            IS_USE_TENANT_DOMAIN_SUBJECT CHAR(1) DEFAULT '1',
            IS_USE_USER_DOMAIN_SUBJECT CHAR(1) DEFAULT '1',
            ENABLE_AUTHORIZATION CHAR(1) DEFAULT '0',
            SUBJECT_CLAIM_URI VARCHAR (512),
            IS_SAAS_APP CHAR(1) DEFAULT '0',
            IS_DUMB_MODE CHAR(1) DEFAULT '0',
            UUID CHAR(36),
            IMAGE_URL VARCHAR(1024),
            ACCESS_URL VARCHAR(1024),
            IS_DISCOVERABLE CHAR(1) DEFAULT '0',
            PRIMARY KEY (ID))
/
ALTER TABLE SP_APP ADD CONSTRAINT APPLICATION_NAME_CONSTRAINT UNIQUE(APP_NAME, TENANT_ID)
/
ALTER TABLE SP_APP ADD CONSTRAINT APPLICATION_UUID_CONSTRAINT UNIQUE(UUID)
/
CREATE TABLE SP_METADATA (
            ID INTEGER,
            SP_ID INTEGER,
            NAME VARCHAR(255) NOT NULL,
            VALUE VARCHAR(255) NOT NULL,
            DISPLAY_NAME VARCHAR(255),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (ID),
            CONSTRAINT SP_METADATA_CONSTRAINT UNIQUE (SP_ID, NAME),
            FOREIGN KEY (SP_ID) REFERENCES SP_APP(ID))
/
CREATE TABLE SP_INBOUND_AUTH (
            ID INTEGER,
            TENANT_ID INTEGER NOT NULL,
            INBOUND_AUTH_KEY VARCHAR (255),
            INBOUND_AUTH_TYPE VARCHAR (255) NOT NULL,
            INBOUND_CONFIG_TYPE VARCHAR (255) NOT NULL,
            PROP_NAME VARCHAR (255),
            PROP_VALUE VARCHAR (1024) ,
            APP_ID INTEGER NOT NULL,
            PRIMARY KEY (ID))
/
ALTER TABLE SP_INBOUND_AUTH ADD CONSTRAINT APPLICATION_ID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_AUTH_STEP (
            ID INTEGER,
            TENANT_ID INTEGER NOT NULL,
	     	STEP_ORDER INTEGER DEFAULT 1,
  APP_ID INTEGER NOT NULL,
            IS_SUBJECT_STEP CHAR(1) DEFAULT '0',
            IS_ATTRIBUTE_STEP CHAR(1) DEFAULT '0',
            PRIMARY KEY (ID))
/
ALTER TABLE SP_AUTH_STEP ADD CONSTRAINT APPLICATION_ID_CONSTRAINT_STEP FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_FEDERATED_IDP (
            ID INTEGER NOT NULL,
            TENANT_ID INTEGER NOT NULL,
            AUTHENTICATOR_ID INTEGER NOT NULL,
            PRIMARY KEY (ID, AUTHENTICATOR_ID))
/
ALTER TABLE SP_FEDERATED_IDP ADD CONSTRAINT STEP_ID_CONSTRAINT FOREIGN KEY (ID) REFERENCES SP_AUTH_STEP (ID)
/
CREATE TABLE SP_CLAIM_DIALECT (
	    	ID INTEGER,
	    	TENANT_ID INTEGER NOT NULL,
	    	SP_DIALECT VARCHAR (512) NOT NULL,
	   		APP_ID INTEGER NOT NULL,
        PRIMARY KEY (ID))
/
ALTER TABLE SP_CLAIM_DIALECT ADD CONSTRAINT DIALECTID_APPID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_CLAIM_MAPPING (
	    	ID INTEGER,
	    	TENANT_ID INTEGER NOT NULL,
	    	IDP_CLAIM VARCHAR (512) NOT NULL ,
            SP_CLAIM VARCHAR (512) NOT NULL ,
	   		APP_ID INTEGER NOT NULL,
	    	IS_REQUESTED VARCHAR(128) DEFAULT '0',
		IS_MANDATORY VARCHAR(128) DEFAULT '0',
			DEFAULT_VALUE VARCHAR(255),
            PRIMARY KEY (ID))
/
ALTER TABLE SP_CLAIM_MAPPING ADD CONSTRAINT CLAIMID_APPID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_ROLE_MAPPING (
	    	ID INTEGER,
	    	TENANT_ID INTEGER NOT NULL,
	    	IDP_ROLE VARCHAR (255) NOT NULL ,
            SP_ROLE VARCHAR (255) NOT NULL ,
	    	APP_ID INTEGER NOT NULL,
            PRIMARY KEY (ID))
/
ALTER TABLE SP_ROLE_MAPPING ADD CONSTRAINT ROLEID_APPID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_REQ_PATH_AUTHENTICATOR (
	    	ID INTEGER,
	    	TENANT_ID INTEGER NOT NULL,
	    	AUTHENTICATOR_NAME VARCHAR (255) NOT NULL ,
	    	APP_ID INTEGER NOT NULL,
            PRIMARY KEY (ID))
/
ALTER TABLE SP_REQ_PATH_AUTHENTICATOR ADD CONSTRAINT REQ_AUTH_APPID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_PROVISIONING_CONNECTOR (
	    	ID INTEGER,
	    	TENANT_ID INTEGER NOT NULL,
            IDP_NAME VARCHAR (255) NOT NULL ,
	    	CONNECTOR_NAME VARCHAR (255) NOT NULL ,
	    	APP_ID INTEGER NOT NULL,
	    	IS_JIT_ENABLED CHAR(1) DEFAULT '0',
		    BLOCKING CHAR(1) DEFAULT '0',
		    RULE_ENABLED CHAR(1) DEFAULT '0' NOT NULL,
            PRIMARY KEY (ID))
/
ALTER TABLE SP_PROVISIONING_CONNECTOR ADD CONSTRAINT PRO_CONNECTOR_APPID_CONSTRAINT FOREIGN KEY (APP_ID) REFERENCES SP_APP (ID)
/
CREATE TABLE SP_AUTH_SCRIPT (
  ID         INTEGER      NOT NULL,
  TENANT_ID  INTEGER      NOT NULL,
  APP_ID     INTEGER      NOT NULL,
  TYPE       VARCHAR(255) NOT NULL,
  CONTENT    BLOB    DEFAULT NULL,
  IS_ENABLED CHAR(1) DEFAULT '0',
  PRIMARY KEY (ID)
)
/
CREATE TABLE SP_TEMPLATE (
  ID         INTEGER      NOT NULL,
  TENANT_ID  INTEGER      NOT NULL,
  NAME       VARCHAR(255) NOT NULL,
  DESCRIPTION  VARCHAR(1023),
  CONTENT BLOB DEFAULT NULL,
  PRIMARY KEY (ID),
  CONSTRAINT SP_TEMPLATE_CONSTRAINT UNIQUE (TENANT_ID, NAME)
)
/
CREATE TABLE IDN_AUTH_WAIT_STATUS (
  ID              INTEGER       NOT NULL,
  TENANT_ID       INTEGER       NOT NULL,
  LONG_WAIT_KEY   VARCHAR(255)  NOT NULL,
  WAIT_STATUS     CHAR(1)       DEFAULT '1',
  TIME_CREATED    TIMESTAMP,
  EXPIRE_TIME     TIMESTAMP,
  PRIMARY KEY (ID),
  CONSTRAINT IDN_AUTH_WAIT_STATUS_KEY UNIQUE (LONG_WAIT_KEY)
)
/
CREATE TABLE IDP (
  ID                         INTEGER,
  TENANT_ID                  INTEGER,
  NAME                       VARCHAR(254) NOT NULL,
  IS_ENABLED                 CHAR(1) DEFAULT '1',
  IS_PRIMARY                 CHAR(1) DEFAULT '0',
  HOME_REALM_ID              VARCHAR(254),
  IMAGE                      BLOB,
  CERTIFICATE                BLOB,
  ALIAS                      VARCHAR(254),
  INBOUND_PROV_ENABLED       CHAR(1) DEFAULT '0',
  INBOUND_PROV_USER_STORE_ID VARCHAR(254),
  USER_CLAIM_URI             VARCHAR(254),
  ROLE_CLAIM_URI             VARCHAR(254),
  DESCRIPTION                VARCHAR(1024),
  DEFAULT_AUTHENTICATOR_NAME VARCHAR(254),
  DEFAULT_PRO_CONNECTOR_NAME VARCHAR(254),
  PROVISIONING_ROLE          VARCHAR(128),
  IS_FEDERATION_HUB          CHAR(1) DEFAULT '0',
  IS_LOCAL_CLAIM_DIALECT     CHAR(1) DEFAULT '0',
  DISPLAY_NAME               VARCHAR(255),
  IMAGE_URL                  VARCHAR(1024),
  UUID                       CHAR(36) NOT NULL,
  PRIMARY KEY (ID),
  UNIQUE (TENANT_ID, NAME),
  UNIQUE (UUID)
)
/
CREATE TABLE IDP_ROLE (
			ID INTEGER,
			IDP_ID INTEGER,
			TENANT_ID INTEGER,
			ROLE VARCHAR(254),
			PRIMARY KEY (ID),
			UNIQUE (IDP_ID, ROLE),
			FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDP_ROLE_MAPPING (
			ID INTEGER,
			IDP_ROLE_ID INTEGER,
			TENANT_ID INTEGER,
			USER_STORE_ID VARCHAR (253),
			LOCAL_ROLE VARCHAR(253),
			PRIMARY KEY (ID),
			UNIQUE (IDP_ROLE_ID, TENANT_ID, USER_STORE_ID, LOCAL_ROLE),
			FOREIGN KEY (IDP_ROLE_ID) REFERENCES IDP_ROLE(ID))
/
CREATE TABLE IDP_CLAIM (
			ID INTEGER,
			IDP_ID INTEGER,
			TENANT_ID INTEGER,
			CLAIM VARCHAR(254),
			PRIMARY KEY (ID),
			UNIQUE (IDP_ID, CLAIM),
			FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDP_CLAIM_MAPPING (
			ID INTEGER,
			IDP_CLAIM_ID INTEGER,
			TENANT_ID INTEGER,
			LOCAL_CLAIM VARCHAR(253),
		    DEFAULT_VALUE VARCHAR(255),
	    	IS_REQUESTED VARCHAR(128) DEFAULT '0',
			PRIMARY KEY (ID),
			UNIQUE (IDP_CLAIM_ID, TENANT_ID, LOCAL_CLAIM),
			FOREIGN KEY (IDP_CLAIM_ID) REFERENCES IDP_CLAIM(ID))
/
CREATE TABLE IDP_AUTHENTICATOR (
            ID INTEGER,
            TENANT_ID INTEGER,
            IDP_ID INTEGER,
            NAME VARCHAR(255) NOT NULL,
            IS_ENABLED CHAR (1) DEFAULT '1',
            DISPLAY_NAME VARCHAR(255),
            PRIMARY KEY (ID),
            UNIQUE (TENANT_ID, IDP_ID, NAME),
            FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDP_METADATA (
            ID INTEGER,
            IDP_ID INTEGER,
            NAME VARCHAR(255) NOT NULL,
            VALUE VARCHAR(255) NOT NULL,
            DISPLAY_NAME VARCHAR(255),
            TENANT_ID INTEGER DEFAULT -1,
            PRIMARY KEY (ID),
            CONSTRAINT IDP_METADATA_CONSTRAINT UNIQUE (IDP_ID, NAME),
            FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDP_AUTHENTICATOR_PROPERTY (
            ID INTEGER,
            TENANT_ID INTEGER,
            AUTHENTICATOR_ID INTEGER,
            PROPERTY_KEY VARCHAR(255) NOT NULL,
            PROPERTY_VALUE VARCHAR(2047),
            IS_SECRET CHAR (1) DEFAULT '0',
            PRIMARY KEY (ID),
            UNIQUE (TENANT_ID, AUTHENTICATOR_ID, PROPERTY_KEY),
            FOREIGN KEY (AUTHENTICATOR_ID) REFERENCES IDP_AUTHENTICATOR(ID))
/
CREATE TABLE IDP_PROVISIONING_CONFIG (
            ID INTEGER,
            TENANT_ID INTEGER,
            IDP_ID INTEGER,
            PROVISIONING_CONNECTOR_TYPE VARCHAR(255) NOT NULL,
            IS_ENABLED CHAR (1) DEFAULT '0',
            IS_BLOCKING CHAR (1) DEFAULT '0',
            IS_RULES_ENABLED CHAR (1) DEFAULT '0',
            PRIMARY KEY (ID),
            UNIQUE (TENANT_ID, IDP_ID, PROVISIONING_CONNECTOR_TYPE),
            FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDP_PROV_CONFIG_PROPERTY (
            ID INTEGER,
            TENANT_ID INTEGER,
            PROVISIONING_CONFIG_ID INTEGER,
            PROPERTY_KEY VARCHAR(255) NOT NULL,
            PROPERTY_VALUE VARCHAR(2048),
            PROPERTY_BLOB_VALUE BLOB,
            PROPERTY_TYPE CHAR(32) NOT NULL,
            IS_SECRET CHAR (1) DEFAULT '0',
            PRIMARY KEY (ID),
            UNIQUE (TENANT_ID, PROVISIONING_CONFIG_ID, PROPERTY_KEY),
            FOREIGN KEY (PROVISIONING_CONFIG_ID) REFERENCES IDP_PROVISIONING_CONFIG(ID))
/
CREATE TABLE IDP_PROVISIONING_ENTITY (
            ID INTEGER,
            PROVISIONING_CONFIG_ID INTEGER,
            ENTITY_TYPE VARCHAR(255) NOT NULL,
            ENTITY_LOCAL_USERSTORE VARCHAR(255) NOT NULL,
            ENTITY_NAME VARCHAR(255) NOT NULL,
            ENTITY_VALUE VARCHAR(255),
            TENANT_ID INTEGER,
            ENTITY_LOCAL_ID VARCHAR(255),
            PRIMARY KEY (ID),
            UNIQUE (ENTITY_TYPE, TENANT_ID, ENTITY_LOCAL_USERSTORE, ENTITY_NAME, PROVISIONING_CONFIG_ID),
            UNIQUE (PROVISIONING_CONFIG_ID, ENTITY_TYPE, ENTITY_VALUE),
            FOREIGN KEY (PROVISIONING_CONFIG_ID) REFERENCES IDP_PROVISIONING_CONFIG(ID))
/
CREATE TABLE IDP_LOCAL_CLAIM (
            ID INTEGER,
            TENANT_ID INTEGER,
            IDP_ID INTEGER,
            CLAIM_URI VARCHAR(255) NOT NULL,
            DEFAULT_VALUE VARCHAR(255),
	        IS_REQUESTED VARCHAR(128) DEFAULT '0',
            PRIMARY KEY (ID),
            UNIQUE (TENANT_ID, IDP_ID, CLAIM_URI),
            FOREIGN KEY (IDP_ID) REFERENCES IDP(ID))
/
CREATE TABLE IDN_ASSOCIATED_ID (
            ID INTEGER,
	    IDP_USER_ID VARCHAR(255) NOT NULL,
            TENANT_ID INTEGER DEFAULT -1234,
	    IDP_ID INTEGER NOT NULL,
            DOMAIN_NAME VARCHAR(255) NOT NULL,
 	    USER_NAME VARCHAR(255) NOT NULL,
            ASSOCIATION_ID CHAR(36) NOT NULL,
	    PRIMARY KEY (ID),
            UNIQUE(IDP_USER_ID, TENANT_ID, IDP_ID),
            FOREIGN KEY (IDP_ID) REFERENCES IDP(ID)
)
/
CREATE TABLE IDN_USER_ACCOUNT_ASSOCIATION (
            ASSOCIATION_KEY VARCHAR(255) NOT NULL,
            TENANT_ID INTEGER,
            DOMAIN_NAME VARCHAR(255) NOT NULL,
            USER_NAME VARCHAR(255) NOT NULL,
            PRIMARY KEY (TENANT_ID, DOMAIN_NAME, USER_NAME))
/
CREATE TABLE FIDO_DEVICE_STORE (
        TENANT_ID INTEGER,
        DOMAIN_NAME VARCHAR(255) NOT NULL,
        USER_NAME VARCHAR(45) NOT NULL,
	TIME_REGISTERED TIMESTAMP,
        KEY_HANDLE VARCHAR(200) NOT NULL,
        DEVICE_DATA VARCHAR(2048) NOT NULL,
      PRIMARY KEY (TENANT_ID, DOMAIN_NAME, USER_NAME, KEY_HANDLE))
/

CREATE TABLE FIDO2_DEVICE_STORE (
        TENANT_ID INTEGER,
        DOMAIN_NAME VARCHAR(255) NOT NULL,
        USER_NAME VARCHAR(45) NOT NULL,
        TIME_REGISTERED TIMESTAMP,
        USER_HANDLE VARCHAR(64) NOT NULL,
        CREDENTIAL_ID VARCHAR(200) NOT NULL,
        PUBLIC_KEY_COSE VARCHAR(1024) NOT NULL,
        SIGNATURE_COUNT NUMBER(19),
        USER_IDENTITY VARCHAR(512) NOT NULL,
        DISPLAY_NAME VARCHAR(255),
        IS_USERNAMELESS_SUPPORTED CHAR(1) DEFAULT '0',
      PRIMARY KEY (CREDENTIAL_ID, USER_HANDLE))
/

CREATE TABLE WF_REQUEST (
    UUID VARCHAR2 (45),
    CREATED_BY VARCHAR2 (255),
    TENANT_ID INTEGER DEFAULT -1,
    OPERATION_TYPE VARCHAR2 (50),
    CREATED_AT TIMESTAMP,
    UPDATED_AT TIMESTAMP,
    STATUS VARCHAR2 (30),
    REQUEST BLOB,
    PRIMARY KEY (UUID))
/

CREATE TABLE WF_BPS_PROFILE (
    PROFILE_NAME VARCHAR2(45),
    HOST_URL_MANAGER VARCHAR2(255),
    HOST_URL_WORKER VARCHAR2(255),
    USERNAME VARCHAR2(45),
    PASSWORD VARCHAR2(1023),
    CALLBACK_HOST VARCHAR2 (45),
    TENANT_ID INTEGER DEFAULT -1,
    PRIMARY KEY (PROFILE_NAME, TENANT_ID))
/

CREATE TABLE WF_WORKFLOW(
    ID VARCHAR2 (45),
    WF_NAME VARCHAR2 (45),
    DESCRIPTION VARCHAR2 (255),
    TEMPLATE_ID VARCHAR2 (45),
    IMPL_ID VARCHAR2 (45),
    TENANT_ID INTEGER DEFAULT -1,
    PRIMARY KEY (ID))
/

CREATE TABLE WF_WORKFLOW_ASSOCIATION(
    ID INTEGER,
    ASSOC_NAME VARCHAR2 (45),
    EVENT_ID VARCHAR2(45),
    ASSOC_CONDITION VARCHAR2 (2000),
    WORKFLOW_ID VARCHAR2 (45),
    IS_ENABLED CHAR (1) DEFAULT '1',
    TENANT_ID INTEGER DEFAULT -1,
    PRIMARY KEY(ID),
    FOREIGN KEY (WORKFLOW_ID) REFERENCES WF_WORKFLOW(ID))
/
CREATE TABLE WF_WORKFLOW_CONFIG_PARAM(
    WORKFLOW_ID VARCHAR2 (45),
    PARAM_NAME VARCHAR2 (45),
    PARAM_VALUE VARCHAR2 (1000),
    PARAM_QNAME VARCHAR2 (45),
    PARAM_HOLDER VARCHAR2 (45),
    TENANT_ID INTEGER DEFAULT -1,
    PRIMARY KEY (WORKFLOW_ID, PARAM_NAME, PARAM_QNAME, PARAM_HOLDER),
    FOREIGN KEY (WORKFLOW_ID) REFERENCES WF_WORKFLOW(ID))
/
CREATE TABLE WF_REQUEST_ENTITY_RELATIONSHIP(
  REQUEST_ID VARCHAR2 (45),
  ENTITY_NAME VARCHAR2 (255),
  ENTITY_TYPE VARCHAR2 (50),
  TENANT_ID INTEGER DEFAULT -1,
  PRIMARY KEY(REQUEST_ID, ENTITY_NAME, ENTITY_TYPE, TENANT_ID),
  FOREIGN KEY (REQUEST_ID) REFERENCES WF_REQUEST(UUID))
/

CREATE TABLE WF_WORKFLOW_REQUEST_RELATION(
  RELATIONSHIP_ID VARCHAR2 (45),
  WORKFLOW_ID VARCHAR2 (45),
  REQUEST_ID VARCHAR2 (45),
  UPDATED_AT TIMESTAMP,
  STATUS VARCHAR (30),
  TENANT_ID INTEGER DEFAULT -1,
  PRIMARY KEY (RELATIONSHIP_ID),
  FOREIGN KEY (WORKFLOW_ID) REFERENCES WF_WORKFLOW(ID),
  FOREIGN KEY (REQUEST_ID) REFERENCES WF_REQUEST(UUID))
/

CREATE TABLE IDN_RECOVERY_DATA (
  USER_NAME VARCHAR2(255) NOT NULL,
  USER_DOMAIN VARCHAR2(127) NOT NULL,
  TENANT_ID INTEGER DEFAULT -1,
  CODE VARCHAR2(255) NOT NULL,
  SCENARIO VARCHAR2(255) NOT NULL,
  STEP VARCHAR2(127) NOT NULL,
  TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
  REMAINING_SETS VARCHAR2(2500) DEFAULT NULL,
  PRIMARY KEY(USER_NAME, USER_DOMAIN, TENANT_ID, SCENARIO,STEP),
  UNIQUE(CODE)
)
/
CREATE TABLE IDN_PASSWORD_HISTORY_DATA (
  ID INTEGER,
  USER_NAME   VARCHAR2(255) NOT NULL,
  USER_DOMAIN VARCHAR2(127) NOT NULL,
  TENANT_ID   INTEGER DEFAULT -1,
  SALT_VALUE  VARCHAR2(255),
  HASH        VARCHAR2(255) NOT NULL,
  TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY(ID),
  UNIQUE (USER_NAME,USER_DOMAIN,TENANT_ID,SALT_VALUE,HASH)
)
/
CREATE TABLE IDN_CLAIM_DIALECT (
  ID INTEGER,
  DIALECT_URI VARCHAR (255) NOT NULL,
  TENANT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  CONSTRAINT DIALECT_URI_CONSTRAINT UNIQUE (DIALECT_URI, TENANT_ID))
/
CREATE TABLE IDN_CLAIM (
  ID INTEGER,
  DIALECT_ID INTEGER NOT NULL,
  CLAIM_URI VARCHAR (255) NOT NULL,
  TENANT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (DIALECT_ID) REFERENCES IDN_CLAIM_DIALECT(ID),
  CONSTRAINT CLAIM_URI_CONSTRAINT UNIQUE (DIALECT_ID, CLAIM_URI, TENANT_ID))
/
CREATE TABLE IDN_CLAIM_MAPPED_ATTRIBUTE (
  ID INTEGER,
  LOCAL_CLAIM_ID INTEGER,
  USER_STORE_DOMAIN_NAME VARCHAR (255) NOT NULL,
  ATTRIBUTE_NAME VARCHAR (255) NOT NULL,
  TENANT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (LOCAL_CLAIM_ID) REFERENCES IDN_CLAIM(ID),
  CONSTRAINT USER_STORE_DOMAIN_CONSTRAINT UNIQUE (LOCAL_CLAIM_ID, USER_STORE_DOMAIN_NAME, TENANT_ID))
/
CREATE TABLE IDN_CLAIM_PROPERTY (
  ID INTEGER,
  LOCAL_CLAIM_ID INTEGER,
  PROPERTY_NAME VARCHAR (255) NOT NULL,
  PROPERTY_VALUE VARCHAR (255) NOT NULL,
  TENANT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (LOCAL_CLAIM_ID) REFERENCES IDN_CLAIM(ID),
  CONSTRAINT PROPERTY_NAME_CONSTRAINT UNIQUE (LOCAL_CLAIM_ID, PROPERTY_NAME, TENANT_ID))
/
CREATE TABLE IDN_CLAIM_MAPPING (
  ID INTEGER,
  EXT_CLAIM_ID INTEGER NOT NULL,
  MAPPED_LOCAL_CLAIM_ID INTEGER NOT NULL,
  TENANT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (EXT_CLAIM_ID) REFERENCES IDN_CLAIM(ID),
  FOREIGN KEY (MAPPED_LOCAL_CLAIM_ID) REFERENCES IDN_CLAIM(ID),
  CONSTRAINT EXT_TO_LOC_MAPPING_CONSTRN UNIQUE (EXT_CLAIM_ID, TENANT_ID))
/
CREATE TABLE IDN_SAML2_ASSERTION_STORE (
  ID INTEGER,
  SAML2_ID  VARCHAR(255) ,
  SAML2_ISSUER  VARCHAR(255) ,
  SAML2_SUBJECT  VARCHAR(255) ,
  SAML2_SESSION_INDEX  VARCHAR(255) ,
  SAML2_AUTHN_CONTEXT_CLASS_REF  VARCHAR(255) ,
  SAML2_ASSERTION  VARCHAR2(4000) ,
  ASSERTION BLOB ,
  PRIMARY KEY (ID))
/
CREATE TABLE IDN_SAML2_ARTIFACT_STORE (
  ID INTEGER,
  SOURCE_ID VARCHAR(255) NOT NULL,
  MESSAGE_HANDLER VARCHAR(255) NOT NULL,
  AUTHN_REQ_DTO BLOB NOT NULL,
  SESSION_ID VARCHAR(255) NOT NULL,
  INIT_TIMESTAMP TIMESTAMP NOT NULL,
  EXP_TIMESTAMP TIMESTAMP NOT NULL,
  ASSERTION_ID VARCHAR(255),
  PRIMARY KEY (ID))
/
CREATE TABLE IDN_OIDC_JTI (
  JWT_ID VARCHAR(255) NOT NULL,
  EXP_TIME TIMESTAMP NOT NULL,
  TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
  PRIMARY KEY (JWT_ID))
/
CREATE TABLE IDN_OIDC_PROPERTY (
  ID INTEGER NOT NULL,
  TENANT_ID  INTEGER,
  CONSUMER_KEY  VARCHAR(255) ,
  PROPERTY_KEY  VARCHAR(255) NOT NULL,
  PROPERTY_VALUE  VARCHAR(2047) ,
  PRIMARY KEY (ID),
  FOREIGN KEY (CONSUMER_KEY) REFERENCES IDN_OAUTH_CONSUMER_APPS(CONSUMER_KEY))
/
CREATE TABLE IDN_OIDC_REQ_OBJECT_REFERENCE (
  ID INTEGER,
  CONSUMER_KEY_ID INTEGER ,
  CODE_ID VARCHAR(255) ,
  TOKEN_ID VARCHAR(255) ,
  SESSION_DATA_KEY VARCHAR(255),
  PRIMARY KEY (ID),
  FOREIGN KEY (CONSUMER_KEY_ID) REFERENCES IDN_OAUTH_CONSUMER_APPS(ID),
  FOREIGN KEY (TOKEN_ID) REFERENCES IDN_OAUTH2_ACCESS_TOKEN(TOKEN_ID),
  FOREIGN KEY (CODE_ID) REFERENCES IDN_OAUTH2_AUTHORIZATION_CODE(CODE_ID))
/
CREATE TABLE IDN_OIDC_REQ_OBJECT_CLAIMS (
  ID INTEGER,
  REQ_OBJECT_ID INTEGER ,
  CLAIM_ATTRIBUTE VARCHAR(255) ,
  ESSENTIAL CHAR(1) DEFAULT '0',
  VALUE VARCHAR(255),
  IS_USERINFO CHAR(1) DEFAULT '0',
  PRIMARY KEY (ID),
  FOREIGN KEY (REQ_OBJECT_ID) REFERENCES IDN_OIDC_REQ_OBJECT_REFERENCE(ID))
/
CREATE TABLE IDN_OIDC_REQ_OBJ_CLAIM_VALUES (
  ID INTEGER,
  REQ_OBJECT_CLAIMS_ID INTEGER,
  CLAIM_VALUES VARCHAR(255),
  PRIMARY KEY (ID),
  FOREIGN KEY (REQ_OBJECT_CLAIMS_ID) REFERENCES IDN_OIDC_REQ_OBJECT_CLAIMS(ID) )
/
CREATE TABLE IDN_CERTIFICATE (
            ID INTEGER,
            NAME VARCHAR(100),
            CERTIFICATE_IN_PEM BLOB,
            TENANT_ID INTEGER DEFAULT 0,
            PRIMARY KEY(ID),
            CONSTRAINT CERTIFICATE_UNIQUE_KEY UNIQUE (NAME, TENANT_ID))
/
CREATE TABLE IDN_OIDC_SCOPE_CLAIM_MAPPING (
  ID INTEGER NOT NULL,
  SCOPE_ID INTEGER NOT NULL,
  EXTERNAL_CLAIM_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (SCOPE_ID) REFERENCES IDN_OAUTH2_SCOPE(SCOPE_ID),
  FOREIGN KEY (EXTERNAL_CLAIM_ID) REFERENCES IDN_CLAIM(ID),
  UNIQUE (SCOPE_ID, EXTERNAL_CLAIM_ID))
/
CREATE TABLE IDN_FUNCTION_LIBRARY (
	NAME VARCHAR(255) NOT NULL,
	DESCRIPTION VARCHAR(1023),
	TYPE VARCHAR(255) NOT NULL,
	TENANT_ID INTEGER NOT NULL,
	DATA BLOB NOT NULL,
	PRIMARY KEY (TENANT_ID,NAME)
)
/
CREATE TABLE IDN_OAUTH2_CIBA_AUTH_CODE (
    AUTH_CODE_KEY CHAR (36),
    AUTH_REQ_ID CHAR (36),
    ISSUED_TIME TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    CONSUMER_KEY VARCHAR(255),
    LAST_POLLED_TIME TIMESTAMP,
    POLLING_INTERVAL INTEGER,
    EXPIRES_IN  INTEGER,
    AUTHENTICATED_USER_NAME VARCHAR(255),
    USER_STORE_DOMAIN VARCHAR(100),
    TENANT_ID INTEGER,
    AUTH_REQ_STATUS VARCHAR (100) DEFAULT 'REQUESTED',
    IDP_ID INTEGER,
    CONSTRAINT AUTH_REQ_ID_CONSTRAINT UNIQUE(AUTH_REQ_ID),
    PRIMARY KEY (AUTH_CODE_KEY),
    FOREIGN KEY (CONSUMER_KEY) REFERENCES IDN_OAUTH_CONSUMER_APPS(CONSUMER_KEY))
/
CREATE TABLE IDN_OAUTH2_CIBA_REQUEST_SCOPES (
    AUTH_CODE_KEY CHAR (36),
    SCOPE VARCHAR (255),
    FOREIGN KEY (AUTH_CODE_KEY) REFERENCES IDN_OAUTH2_CIBA_AUTH_CODE(AUTH_CODE_KEY))
/
CREATE TABLE IDN_FED_AUTH_SESSION_MAPPING (
	IDP_SESSION_ID VARCHAR(255) NOT NULL,
	SESSION_ID VARCHAR(255) NOT NULL,
	IDP_NAME VARCHAR(255) NOT NULL,
	AUTHENTICATOR_ID VARCHAR(255),
	PROTOCOL_TYPE VARCHAR(255),
	TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP NOT NULL,
	PRIMARY KEY (IDP_SESSION_ID)
)
/

-- --------------------------- INDEX CREATION -----------------------------
-- IDN_OAUTH2_ACCESS_TOKEN --
CREATE INDEX IDX_TC ON IDN_OAUTH2_ACCESS_TOKEN(TIME_CREATED)
/
CREATE INDEX IDX_ATH ON IDN_OAUTH2_ACCESS_TOKEN(ACCESS_TOKEN_HASH)
/
CREATE INDEX IDX_AT_CK_AU ON IDN_OAUTH2_ACCESS_TOKEN(CONSUMER_KEY_ID, AUTHZ_USER, TOKEN_STATE, USER_TYPE)
/
CREATE INDEX IDX_AT_TI_UD ON IDN_OAUTH2_ACCESS_TOKEN(AUTHZ_USER, TENANT_ID, TOKEN_STATE, USER_DOMAIN)
/
CREATE INDEX IDX_AT_AU_TID_UD_TS_CKID ON IDN_OAUTH2_ACCESS_TOKEN(AUTHZ_USER, TENANT_ID, USER_DOMAIN, TOKEN_STATE, CONSUMER_KEY_ID)
/
CREATE INDEX IDX_AT_AT ON IDN_OAUTH2_ACCESS_TOKEN(ACCESS_TOKEN)
/
CREATE INDEX IDX_AT_AU_CKID_TS_UT ON IDN_OAUTH2_ACCESS_TOKEN(AUTHZ_USER, CONSUMER_KEY_ID, TOKEN_STATE, USER_TYPE)
/
CREATE INDEX IDX_AT_RTH ON IDN_OAUTH2_ACCESS_TOKEN(REFRESH_TOKEN_HASH)
/
CREATE INDEX IDX_AT_RT ON IDN_OAUTH2_ACCESS_TOKEN(REFRESH_TOKEN)
/
CREATE INDEX IDX_AT_CKID_AU_TID_UD_TSH_TS ON IDN_OAUTH2_ACCESS_TOKEN(CONSUMER_KEY_ID, AUTHZ_USER, TENANT_ID, USER_DOMAIN, TOKEN_SCOPE_HASH, TOKEN_STATE)
/

-- IDN_OAUTH2_AUTHORIZATION_CODE --
CREATE INDEX IDX_AUTHORIZATION_CODE_HASH ON IDN_OAUTH2_AUTHORIZATION_CODE (AUTHORIZATION_CODE_HASH, CONSUMER_KEY_ID)
/
CREATE INDEX IDX_AUTHORIZATION_CODE_AU_TI ON IDN_OAUTH2_AUTHORIZATION_CODE (AUTHZ_USER, TENANT_ID, USER_DOMAIN, STATE)
/
CREATE INDEX IDX_AC_CKID ON IDN_OAUTH2_AUTHORIZATION_CODE(CONSUMER_KEY_ID)
/
CREATE INDEX IDX_AC_TID ON IDN_OAUTH2_AUTHORIZATION_CODE(TOKEN_ID)
/
CREATE INDEX IDX_AC_AC_CKID ON IDN_OAUTH2_AUTHORIZATION_CODE(AUTHORIZATION_CODE, CONSUMER_KEY_ID)
/

-- IDN_SCIM_GROUP --
CREATE INDEX IDX_IDN_SCIM_GROUP_TI_RN ON IDN_SCIM_GROUP (TENANT_ID, ROLE_NAME)
/
CREATE INDEX IDX_IDN_SCIM_GROUP_TI_RN_AN ON IDN_SCIM_GROUP (TENANT_ID, ROLE_NAME, ATTR_NAME)
/

-- IDN_AUTH_SESSION_STORE --
CREATE INDEX IDX_IDN_AUTH_SESSION_TIME ON IDN_AUTH_SESSION_STORE (TIME_CREATED)
/

-- IDN_AUTH_TEMP_SESSION_STORE --
CREATE INDEX IDX_IDN_AUTH_TMP_SESSION_TIME ON IDN_AUTH_TEMP_SESSION_STORE (TIME_CREATED)
/

-- IDN_OAUTH2_SCOPE --
CREATE INDEX IDX_SC_TID ON IDN_OAUTH2_SCOPE(TENANT_ID)
/
CREATE INDEX IDX_SC_N_TID ON IDN_OAUTH2_SCOPE(NAME, TENANT_ID)
/

-- IDN_OAUTH2_SCOPE_BINDING --
CREATE INDEX IDX_SB_SCPID ON IDN_OAUTH2_SCOPE_BINDING(SCOPE_ID)
/

-- IDN_OIDC_REQ_OBJECT_REFERENCE --
CREATE INDEX IDX_OROR_TID ON IDN_OIDC_REQ_OBJECT_REFERENCE(TOKEN_ID)
/

-- IDN_OAUTH2_ACCESS_TOKEN_SCOPE --
CREATE INDEX IDX_ATS_TID ON IDN_OAUTH2_ACCESS_TOKEN_SCOPE(TOKEN_ID)
/

-- IDN_AUTH_USER --
CREATE INDEX IDX_AUTH_USER_UN_TID_DN ON IDN_AUTH_USER (USER_NAME, TENANT_ID, DOMAIN_NAME)
/
CREATE INDEX IDX_AUTH_USER_DN_TOD ON IDN_AUTH_USER (DOMAIN_NAME, TENANT_ID)
/

-- IDN_AUTH_USER_SESSION_MAPPING --
CREATE INDEX IDX_USER_ID ON IDN_AUTH_USER_SESSION_MAPPING (USER_ID)
/
CREATE INDEX IDX_SESSION_ID ON IDN_AUTH_USER_SESSION_MAPPING (SESSION_ID)
/

-- IDN_OAUTH_CONSUMER_APPS --
CREATE INDEX IDX_OCA_UM_TID_UD_APN ON IDN_OAUTH_CONSUMER_APPS(USERNAME,TENANT_ID,USER_DOMAIN, APP_NAME)
/

-- IDX_SPI_APP --
CREATE INDEX IDX_SPI_APP ON SP_INBOUND_AUTH(APP_ID)
/

-- IDN_OIDC_PROPERTY --
CREATE INDEX IDX_IOP_TID_CK ON IDN_OIDC_PROPERTY(TENANT_ID,CONSUMER_KEY)
/

-- IDN_FIDO2_PROPERTY --
CREATE INDEX IDX_FIDO2_STR ON FIDO2_DEVICE_STORE(USER_NAME, TENANT_ID, DOMAIN_NAME, CREDENTIAL_ID, USER_HANDLE)
/

-- IDN_ASSOCIATED_ID --
CREATE INDEX IDX_AI_DN_UN_AI ON IDN_ASSOCIATED_ID(DOMAIN_NAME, USER_NAME, ASSOCIATION_ID)
/

-- IDN_OAUTH2_TOKEN_BINDING --
CREATE INDEX IDX_IDN_AUTH_BIND ON IDN_OAUTH2_TOKEN_BINDING (TOKEN_BINDING_REF)
/

-- IDN_FED_AUTH_SESSION_MAPPING --
CREATE INDEX IDX_FEDERATED_AUTH_SESSION_ID ON IDN_FED_AUTH_SESSION_MAPPING (SESSION_ID)
/

-- End of IDENTITY Tables--

-- UMA tables --
CREATE TABLE IDN_UMA_RESOURCE (
  ID INTEGER,
  RESOURCE_ID VARCHAR2(255),
  RESOURCE_NAME VARCHAR2(255),
  TIME_CREATED TIMESTAMP NOT NULL,
  RESOURCE_OWNER_NAME VARCHAR2(255),
  CLIENT_ID VARCHAR2(255),
  TENANT_ID INTEGER DEFAULT -1234,
  USER_DOMAIN VARCHAR2(50),
  PRIMARY KEY (ID)
)
/
CREATE INDEX IDX_RID ON IDN_UMA_RESOURCE (RESOURCE_ID)
/

CREATE INDEX IDX_USER ON IDN_UMA_RESOURCE (RESOURCE_OWNER_NAME, USER_DOMAIN)
/

CREATE TABLE IDN_UMA_RESOURCE_META_DATA (
  ID INTEGER,
  RESOURCE_IDENTITY INTEGER NOT NULL,
  PROPERTY_KEY VARCHAR2(40),
  PROPERTY_VALUE VARCHAR2(255),
  PRIMARY KEY (ID),
  FOREIGN KEY (RESOURCE_IDENTITY) REFERENCES IDN_UMA_RESOURCE (ID)
)
/
CREATE TABLE IDN_UMA_RESOURCE_SCOPE (
  ID INTEGER,
  RESOURCE_IDENTITY INTEGER NOT NULL,
  SCOPE_NAME VARCHAR2(255),
  PRIMARY KEY (ID),
  FOREIGN KEY (RESOURCE_IDENTITY) REFERENCES IDN_UMA_RESOURCE (ID)
)
/
CREATE INDEX IDX_RS ON IDN_UMA_RESOURCE_SCOPE (SCOPE_NAME)
/
CREATE TABLE IDN_UMA_PERMISSION_TICKET (
  ID INTEGER,
  PT VARCHAR2(255) NOT NULL,
  TIME_CREATED TIMESTAMP NOT NULL,
  EXPIRY_TIME TIMESTAMP NOT NULL,
  TICKET_STATE VARCHAR2(25) DEFAULT 'ACTIVE',
  TENANT_ID INTEGER DEFAULT -1234,
  PRIMARY KEY (ID)
)
/
CREATE INDEX IDX_PT ON IDN_UMA_PERMISSION_TICKET (PT)
/

CREATE TABLE IDN_UMA_PT_RESOURCE (
  ID INTEGER,
  PT_RESOURCE_ID INTEGER NOT NULL,
  PT_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (PT_ID) REFERENCES IDN_UMA_PERMISSION_TICKET (ID),
  FOREIGN KEY (PT_RESOURCE_ID) REFERENCES IDN_UMA_RESOURCE (ID)
)
/
CREATE TABLE IDN_UMA_PT_RESOURCE_SCOPE (
  ID INTEGER,
  PT_RESOURCE_ID INTEGER NOT NULL,
  PT_SCOPE_ID INTEGER NOT NULL,
  PRIMARY KEY (ID),
  FOREIGN KEY (PT_RESOURCE_ID) REFERENCES IDN_UMA_PT_RESOURCE (ID),
  FOREIGN KEY (PT_SCOPE_ID) REFERENCES IDN_UMA_RESOURCE_SCOPE (ID)
)
/
-- Start of CONSENT-MGT Tables --

CREATE TABLE CM_PII_CATEGORY (
  ID           INTEGER,
  NAME         VARCHAR2(255) NOT NULL,
  DESCRIPTION  VARCHAR2(1023),
  DISPLAY_NAME VARCHAR(255),
  IS_SENSITIVE INTEGER       NOT NULL,
  TENANT_ID    INTEGER DEFAULT -1234,
  CONSTRAINT PII_CATEGORY_CONSTRAINT UNIQUE (NAME, TENANT_ID),
  PRIMARY KEY (ID)
)
/
CREATE TABLE CM_RECEIPT (
  CONSENT_RECEIPT_ID  VARCHAR2(255) NOT NULL,
  VERSION             VARCHAR2(255) NOT NULL,
  JURISDICTION        VARCHAR2(255) NOT NULL,
  CONSENT_TIMESTAMP   TIMESTAMP     NOT NULL,
  COLLECTION_METHOD   VARCHAR2(255) NOT NULL,
  LANGUAGE            VARCHAR2(255) NOT NULL,
  PII_PRINCIPAL_ID    VARCHAR2(255) NOT NULL,
  PRINCIPAL_TENANT_ID INTEGER DEFAULT -1234,
  POLICY_URL          VARCHAR2(255) NOT NULL,
  STATE               VARCHAR2(255) NOT NULL,
  PII_CONTROLLER      VARCHAR2(2048) NOT NULL,
  PRIMARY KEY (CONSENT_RECEIPT_ID)
)
/
CREATE TABLE CM_PURPOSE (
  ID            INTEGER,
  NAME          VARCHAR2(255) NOT NULL,
  DESCRIPTION   VARCHAR2(1023),
  PURPOSE_GROUP VARCHAR2(255) NOT NULL,
  GROUP_TYPE    VARCHAR2(255) NOT NULL,
  TENANT_ID     INTEGER DEFAULT -1234,
  CONSTRAINT PURPOSE_CONSTRAINT UNIQUE (NAME, TENANT_ID, PURPOSE_GROUP, GROUP_TYPE),
  PRIMARY KEY (ID)
)
/
CREATE TABLE CM_PURPOSE_CATEGORY (
  ID          INTEGER,
  NAME        VARCHAR2(255) NOT NULL,
  DESCRIPTION VARCHAR2(1023),
  TENANT_ID   INTEGER DEFAULT -1234,
  CONSTRAINT PURPOSE_CATEGORY_CONSTRAINT UNIQUE (NAME, TENANT_ID),
  PRIMARY KEY (ID)
)
/
CREATE TABLE CM_RECEIPT_SP_ASSOC (
  ID                 INTEGER,
  CONSENT_RECEIPT_ID VARCHAR2(255) NOT NULL,
  SP_NAME            VARCHAR2(255) NOT NULL,
  SP_DISPLAY_NAME    VARCHAR(255),
  SP_DESCRIPTION     VARCHAR(255),
  SP_TENANT_ID       INTEGER DEFAULT -1234,
  CONSTRAINT RECEIPT_SP_ASSOC_CONSTRAINT UNIQUE (CONSENT_RECEIPT_ID, SP_NAME, SP_TENANT_ID),
  PRIMARY KEY (ID)
)
/
CREATE TABLE CM_SP_PURPOSE_ASSOC (
  ID                     INTEGER,
  RECEIPT_SP_ASSOC       INTEGER       NOT NULL,
  PURPOSE_ID             INTEGER       NOT NULL,
  CONSENT_TYPE           VARCHAR2(255) NOT NULL,
  IS_PRIMARY_PURPOSE     INTEGER       NOT NULL,
  TERMINATION            VARCHAR2(255) NOT NULL,
  THIRD_PARTY_DISCLOSURE INTEGER       NOT NULL,
  THIRD_PARTY_NAME       VARCHAR2(255),
  CONSTRAINT SP_PURPOSE_ASSOC UNIQUE (RECEIPT_SP_ASSOC, PURPOSE_ID),
  PRIMARY KEY (ID)
)
/
CREATE TABLE CM_SP_PURPOSE_PURPOSE_CAT_ASSC (
  SP_PURPOSE_ASSOC_ID INTEGER NOT NULL,
  PURPOSE_CATEGORY_ID INTEGER NOT NULL,
  CONSTRAINT SP_PUS_PS_CAT_ASSOC UNIQUE (SP_PURPOSE_ASSOC_ID,PURPOSE_CATEGORY_ID)
)
/
CREATE TABLE CM_PURPOSE_PII_CAT_ASSOC (
  PURPOSE_ID         INTEGER NOT NULL,
  CM_PII_CATEGORY_ID INTEGER NOT NULL,
  IS_MANDATORY       INTEGER NOT NULL,
  CONSTRAINT PURPOSE_PII_CAT_ASSOC UNIQUE (PURPOSE_ID, CM_PII_CATEGORY_ID)
)
/
CREATE TABLE CM_SP_PURPOSE_PII_CAT_ASSOC (
  SP_PURPOSE_ASSOC_ID INTEGER NOT NULL,
  PII_CATEGORY_ID     INTEGER NOT NULL,
  VALIDITY            VARCHAR(1023),
  CONSTRAINT SP_PURPOSE_PII_CATEGORY_ASSOC UNIQUE (SP_PURPOSE_ASSOC_ID, PII_CATEGORY_ID)
)
/
CREATE TABLE CM_CONSENT_RECEIPT_PROPERTY (
  CONSENT_RECEIPT_ID VARCHAR2(255)  NOT NULL,
  NAME               VARCHAR2(255)  NOT NULL,
  VALUE              VARCHAR2(1023) NOT NULL,
  CONSTRAINT CONSENT_RECEIPT_PROPERTY UNIQUE (CONSENT_RECEIPT_ID, NAME)
)
/
ALTER TABLE CM_RECEIPT_SP_ASSOC
  ADD CONSTRAINT CM_RECEIPT_SP_ASSOC_fk0 FOREIGN KEY (CONSENT_RECEIPT_ID) REFERENCES CM_RECEIPT (CONSENT_RECEIPT_ID)
/
ALTER TABLE CM_SP_PURPOSE_ASSOC
  ADD CONSTRAINT CM_SP_PURPOSE_ASSOC_fk0 FOREIGN KEY (RECEIPT_SP_ASSOC) REFERENCES CM_RECEIPT_SP_ASSOC (ID)
/
ALTER TABLE CM_SP_PURPOSE_ASSOC
  ADD CONSTRAINT CM_SP_PURPOSE_ASSOC_fk1 FOREIGN KEY (PURPOSE_ID) REFERENCES CM_PURPOSE (ID)
/
ALTER TABLE CM_SP_PURPOSE_PURPOSE_CAT_ASSC
  ADD CONSTRAINT CM_SP_P_P_CAT_ASSOC_fk0 FOREIGN KEY (SP_PURPOSE_ASSOC_ID) REFERENCES CM_SP_PURPOSE_ASSOC (ID)
/
ALTER TABLE CM_SP_PURPOSE_PURPOSE_CAT_ASSC
  ADD CONSTRAINT CM_SP_P_P_CAT_ASSOC_fk1 FOREIGN KEY (PURPOSE_CATEGORY_ID) REFERENCES CM_PURPOSE_CATEGORY (ID)
/
ALTER TABLE CM_SP_PURPOSE_PII_CAT_ASSOC
  ADD CONSTRAINT CM_SP_P_PII_CAT_ASSOC_fk0 FOREIGN KEY (SP_PURPOSE_ASSOC_ID) REFERENCES CM_SP_PURPOSE_ASSOC (ID)
/
ALTER TABLE CM_SP_PURPOSE_PII_CAT_ASSOC
  ADD CONSTRAINT CM_SP_P_PII_CAT_ASSOC_fk1 FOREIGN KEY (PII_CATEGORY_ID) REFERENCES CM_PII_CATEGORY (ID)
/
ALTER TABLE CM_CONSENT_RECEIPT_PROPERTY
  ADD CONSTRAINT CM_CONSENT_RECEIPT_PRT_fk0 FOREIGN KEY (CONSENT_RECEIPT_ID) REFERENCES CM_RECEIPT (CONSENT_RECEIPT_ID)
/
-- End of CONSENT-MGT Tables --

-- Start of API-MGT Tables--
CREATE TABLE  AM_SUBSCRIBER (
    SUBSCRIBER_ID INTEGER,
    USER_ID VARCHAR2(50) NOT NULL,
    TENANT_ID INTEGER NOT NULL,
    EMAIL_ADDRESS VARCHAR2(256) NULL,
    DATE_SUBSCRIBED DATE NOT NULL,
    CREATED_BY VARCHAR2(100),
    CREATED_TIME DATE,
    UPDATED_BY VARCHAR2(100),
    UPDATED_TIME DATE,
    PRIMARY KEY (SUBSCRIBER_ID),
    UNIQUE (TENANT_ID,USER_ID)
)
/
-- TODO: Have to add ON UPDATE CASCADE for the FOREIGN KEY(SUBSCRIBER_ID) relation
CREATE TABLE AM_APPLICATION (
    APPLICATION_ID INTEGER,
    NAME VARCHAR2(100),
    SUBSCRIBER_ID INTEGER,
    APPLICATION_TIER VARCHAR2(50) DEFAULT 'Unlimited',
    CALLBACK_URL VARCHAR2(512),
    DESCRIPTION VARCHAR2(512),
    APPLICATION_STATUS VARCHAR2(50) DEFAULT 'APPROVED',
    GROUP_ID VARCHAR2(100),
    CREATED_BY VARCHAR2(100),
    CREATED_TIME DATE,
    UPDATED_BY VARCHAR2(100),
    UPDATED_TIME DATE,
    UUID VARCHAR2(256),
    TOKEN_TYPE VARCHAR2(100),
    FOREIGN KEY(SUBSCRIBER_ID) REFERENCES AM_SUBSCRIBER(SUBSCRIBER_ID),
    PRIMARY KEY(APPLICATION_ID),
    UNIQUE (NAME,SUBSCRIBER_ID),
    UNIQUE (UUID)
)
/
CREATE TABLE AM_MONETIZATION_USAGE (
	ID VARCHAR(100) NOT NULL,
	STATE VARCHAR(50) NOT NULL,
	STATUS VARCHAR(50) NOT NULL,
	STARTED_TIME VARCHAR(50) NOT NULL,
	PUBLISHED_TIME VARCHAR(50) NOT NULL,
	PRIMARY KEY(ID)
)
/
CREATE TABLE AM_API (
    API_ID INTEGER,
    API_PROVIDER VARCHAR2(256),
    API_NAME VARCHAR2(256),
    API_VERSION VARCHAR2(30),
    CONTEXT VARCHAR2(256),
    CONTEXT_TEMPLATE VARCHAR2(256),
    API_TIER VARCHAR(256),
    API_TYPE VARCHAR(10),
    CREATED_BY VARCHAR2(100),
    CREATED_TIME DATE,
    UPDATED_BY VARCHAR2(100),
    UPDATED_TIME DATE,
    REVISIONS_CREATED INTEGER DEFAULT 0,
    PRIMARY KEY(API_ID),
    UNIQUE (API_PROVIDER,API_NAME,API_VERSION)
)
/

CREATE TABLE AM_API_REVISION_METADATA (
    API_UUID VARCHAR(64),
    REVISION_UUID VARCHAR(64),
    API_TIER VARCHAR(128),
    UNIQUE (API_UUID,REVISION_UUID)
)
/

CREATE TABLE AM_API_URL_MAPPING (
    URL_MAPPING_ID INTEGER,
    API_ID INTEGER NOT NULL,
    HTTP_METHOD VARCHAR(20) NULL,
    AUTH_SCHEME VARCHAR(50) NULL,
    URL_PATTERN VARCHAR(512) NULL,
    THROTTLING_TIER varchar(512) DEFAULT NULL,
    MEDIATION_SCRIPT BLOB,
    PRIMARY KEY(URL_MAPPING_ID)
)
/
CREATE TABLE AM_SECURITY_AUDIT_UUID_MAPPING (
    API_ID INTEGER NOT NULL,
    AUDIT_UUID VARCHAR(255) NOT NULL,
    FOREIGN KEY (API_ID) REFERENCES AM_API(API_ID),
    PRIMARY KEY (API_ID)
)
/
CREATE TABLE AM_API_PRODUCT_MAPPING (
	API_PRODUCT_MAPPING_ID INTEGER,
	API_ID INTEGER,
	URL_MAPPING_ID INTEGER,
	FOREIGN KEY (API_ID) REFERENCES AM_API(API_ID),
	FOREIGN KEY (URL_MAPPING_ID) REFERENCES AM_API_URL_MAPPING(URL_MAPPING_ID),
	PRIMARY KEY(API_PRODUCT_MAPPING_ID)
)
/
-- TODO: Have to add ON UPDATE CASCADE for the FOREIGN KEY(SUBSCRIPTION_ID) relation
CREATE TABLE AM_SUBSCRIPTION (
    SUBSCRIPTION_ID INTEGER,
    TIER_ID VARCHAR2(50),
    TIER_ID_PENDING VARCHAR(50),
    API_ID INTEGER,
    LAST_ACCESSED DATE NULL,
    APPLICATION_ID INTEGER,
    SUB_STATUS VARCHAR(50),
    SUBS_CREATE_STATE VARCHAR(50) DEFAULT 'SUBSCRIBE',
    CREATED_BY VARCHAR2(100),
    CREATED_TIME DATE,
    UPDATED_BY VARCHAR2(100),
    UPDATED_TIME DATE,
    UUID VARCHAR2(256),
    FOREIGN KEY(APPLICATION_ID) REFERENCES AM_APPLICATION(APPLICATION_ID),
    FOREIGN KEY(API_ID) REFERENCES AM_API(API_ID),
    PRIMARY KEY (SUBSCRIPTION_ID),
    UNIQUE (UUID)
)
/

--TODO: Have to add ON UPDATE CASCADE for the FOREIGN KEY(APPLICATION_ID) relation
CREATE TABLE AM_APPLICATION_KEY_MAPPING (
    UUID VARCHAR2(512),
    APPLICATION_ID INTEGER,
    CONSUMER_KEY VARCHAR2(512),
    KEY_TYPE VARCHAR2(512) NOT NULL,
    CREATE_MODE VARCHAR2(30) DEFAULT 'CREATED',
    STATE VARCHAR2(30) NOT NULL,
    KEY_MANAGER VARCHAR2(512),
    APP_INFO BLOB DEFAULT NULL,
    FOREIGN KEY(APPLICATION_ID) REFERENCES AM_APPLICATION(APPLICATION_ID),
    PRIMARY KEY(APPLICATION_ID,KEY_TYPE,KEY_MANAGER)
)
/
CREATE TABLE AM_APPLICATION_REGISTRATION (
    REG_ID INTEGER ,
    SUBSCRIBER_ID INTEGER,
    WF_REF VARCHAR2(255) NOT NULL,
    APP_ID INTEGER,
    TOKEN_TYPE VARCHAR2(30),
    TOKEN_SCOPE VARCHAR2(1500) DEFAULT 'default',
    INPUTS VARCHAR2(1024),
    ALLOWED_DOMAINS VARCHAR2(256),
    VALIDITY_PERIOD NUMBER(19),
    KEY_MANAGER VARCHAR2(255) NOT NULL,
    UNIQUE (SUBSCRIBER_ID,APP_ID,TOKEN_TYPE,KEY_MANAGER),
    FOREIGN KEY(SUBSCRIBER_ID) REFERENCES AM_SUBSCRIBER(SUBSCRIBER_ID),
    FOREIGN KEY(APP_ID) REFERENCES AM_APPLICATION(APPLICATION_ID),
    PRIMARY KEY (REG_ID)
)
/
-- TODO: Have to add ON UPDATE CASCADE for the FOREIGN KEY(API_ID) relation
CREATE TABLE AM_API_LC_EVENT (
    EVENT_ID INTEGER,
    API_ID INTEGER NOT NULL,
    PREVIOUS_STATE VARCHAR2(50),
    NEW_STATE VARCHAR2(50) NOT NULL,
    USER_ID VARCHAR2(50) NOT NULL,
    TENANT_ID INTEGER NOT NULL,
    EVENT_DATE DATE NOT NULL,
    FOREIGN KEY(API_ID) REFERENCES AM_API(API_ID),
    PRIMARY KEY (EVENT_ID)
)
/
CREATE TABLE AM_APP_KEY_DOMAIN_MAPPING (
    CONSUMER_KEY VARCHAR(255),
    AUTHZ_DOMAIN VARCHAR(255) DEFAULT 'ALL',
    PRIMARY KEY (CONSUMER_KEY,AUTHZ_DOMAIN)
)
/
CREATE TABLE AM_API_COMMENTS (
    COMMENT_ID VARCHAR2(255) NOT NULL,
    COMMENT_TEXT VARCHAR2(512),
    COMMENTED_USER VARCHAR2(255),
    DATE_COMMENTED DATE NOT NULL,
    API_ID INTEGER,
    FOREIGN KEY(API_ID) REFERENCES AM_API(API_ID),
    PRIMARY KEY (COMMENT_ID)
)
/
CREATE TABLE AM_API_RATINGS (
    RATING_ID VARCHAR2(255) NOT NULL,
    API_ID INTEGER,
    RATING INTEGER,
    SUBSCRIBER_ID INTEGER,
    FOREIGN KEY(API_ID) REFERENCES AM_API(API_ID),
    FOREIGN KEY(SUBSCRIBER_ID) REFERENCES AM_SUBSCRIBER(SUBSCRIBER_ID),
    PRIMARY KEY (RATING_ID)
)
/
CREATE TABLE AM_TIER_PERMISSIONS (
    TIER_PERMISSIONS_ID INTEGER,
    TIER VARCHAR2(50) NOT NULL,
    PERMISSIONS_TYPE VARCHAR2(50) NOT NULL,
    ROLES VARCHAR2(512) NOT NULL,
    TENANT_ID INTEGER NOT NULL,
    PRIMARY KEY(TIER_PERMISSIONS_ID)
)
/
-- Start of IDP-APPLICATION-MGT Tables --

-- End of IDP-APPLICATION-MGT Tables --


CREATE TABLE AM_EXTERNAL_STORES (
    APISTORE_ID INTEGER,
    API_ID INTEGER,
    STORE_ID VARCHAR2(255) NOT NULL,
    STORE_DISPLAY_NAME VARCHAR2(255) NOT NULL,
    STORE_ENDPOINT VARCHAR2(255) NOT NULL,
    STORE_TYPE VARCHAR2(255) NOT NULL,
    LAST_UPDATED_TIME DATE,
    FOREIGN KEY(API_ID) REFERENCES AM_API(API_ID),
    PRIMARY KEY (APISTORE_ID)
)
/

CREATE TABLE AM_WORKFLOWS(
    WF_ID INTEGER,
    WF_REFERENCE VARCHAR(255) NOT NULL,
    WF_TYPE VARCHAR(255) NOT NULL,
    WF_STATUS VARCHAR(255) NOT NULL,
    WF_CREATED_TIME TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    WF_UPDATED_TIME TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    WF_STATUS_DESC VARCHAR(1000),
    TENANT_ID INTEGER,
    TENANT_DOMAIN VARCHAR(255),
    WF_EXTERNAL_REFERENCE VARCHAR(255) NOT NULL UNIQUE,
    PRIMARY KEY (WF_ID)
)
/

CREATE TABLE AM_API_SCOPES (
   API_ID  INTEGER NOT NULL,
   SCOPE_ID  INTEGER NOT NULL,
   FOREIGN KEY (API_ID) REFERENCES AM_API (API_ID),
   FOREIGN KEY (SCOPE_ID) REFERENCES IDN_OAUTH2_SCOPE (SCOPE_ID),
   PRIMARY KEY (API_ID, SCOPE_ID)
)
/
CREATE TABLE AM_API_DEFAULT_VERSION (
            DEFAULT_VERSION_ID NUMBER,
            API_NAME VARCHAR(256) NOT NULL ,
            API_PROVIDER VARCHAR(256) NOT NULL ,
            DEFAULT_API_VERSION VARCHAR(30) ,
            PUBLISHED_DEFAULT_API_VERSION VARCHAR(30) ,
            PRIMARY KEY (DEFAULT_VERSION_ID)
)
/
CREATE INDEX IDX_SUB_APP_ID ON AM_SUBSCRIPTION (APPLICATION_ID, SUBSCRIPTION_ID)
/

CREATE TABLE  AM_ALERT_TYPES (
            ALERT_TYPE_ID INTEGER,
            ALERT_TYPE_NAME VARCHAR(255) NOT NULL ,
	    STAKE_HOLDER VARCHAR(100) NOT NULL,
            PRIMARY KEY (ALERT_TYPE_ID))
/
CREATE TABLE  AM_ALERT_TYPES_VALUES (
            ALERT_TYPE_ID INTEGER,
            USER_NAME VARCHAR(255) NOT NULL ,
	    STAKE_HOLDER VARCHAR(100) NOT NULL ,
            CONSTRAINT AM_ALERT_TYPES_VALUES_CONST UNIQUE (ALERT_TYPE_ID,USER_NAME,STAKE_HOLDER))
/

CREATE TABLE  AM_ALERT_EMAILLIST (
	    EMAIL_LIST_ID INTEGER,
            USER_NAME VARCHAR(255) NOT NULL ,
	    STAKE_HOLDER VARCHAR(100) NOT NULL ,
            CONSTRAINT AM_ALERT_EMAILLIST_CONST UNIQUE (EMAIL_LIST_ID,USER_NAME,STAKE_HOLDER),
             PRIMARY KEY (EMAIL_LIST_ID))
/
CREATE TABLE  AM_ALERT_EMAILLIST_DETAILS (
            EMAIL_LIST_ID INTEGER,
	    EMAIL VARCHAR(255),
            CONSTRAINT AM_ALERT_EMAIL_LIST_DET_CONST UNIQUE (EMAIL_LIST_ID,EMAIL))
/

-- AM Throttling tables --

CREATE TABLE AM_POLICY_SUBSCRIPTION (
            POLICY_ID INTEGER NOT NULL,
            NAME VARCHAR2(512) NOT NULL,
            DISPLAY_NAME VARCHAR2(512) DEFAULT NULL NULL,
            TENANT_ID INTEGER NOT NULL,
            DESCRIPTION VARCHAR2(1024) DEFAULT NULL NULL,
            QUOTA_TYPE VARCHAR2(25) NOT NULL,
            QUOTA INTEGER NOT NULL,
            QUOTA_UNIT VARCHAR2(10) NULL,
            UNIT_TIME INTEGER NOT NULL,
            TIME_UNIT VARCHAR2(25) NOT NULL,
            RATE_LIMIT_COUNT INTEGER DEFAULT NULL NULL,
            RATE_LIMIT_TIME_UNIT VARCHAR2(25) DEFAULT NULL NULL,
            IS_DEPLOYED INTEGER DEFAULT 0 NOT NULL,
	    CUSTOM_ATTRIBUTES BLOB DEFAULT NULL,
	    STOP_ON_QUOTA_REACH INTEGER DEFAULT 0 NOT NULL,
	    BILLING_PLAN VARCHAR2(20),
	          UUID VARCHAR2(256),
            MONETIZATION_PLAN VARCHAR(25) DEFAULT NULL NULL,
            FIXED_RATE VARCHAR(15) DEFAULT NULL NULL,
            BILLING_CYCLE VARCHAR(15) DEFAULT NULL NULL,
            PRICE_PER_REQUEST VARCHAR(15) DEFAULT NULL NULL,
            CURRENCY VARCHAR(15) DEFAULT NULL NULL,
            PRIMARY KEY (POLICY_ID),
            CONSTRAINT SUBSCRIPTION_NAME_TENANT UNIQUE (NAME, TENANT_ID),
            UNIQUE (UUID)
)
/
CREATE TABLE AM_POLICY_APPLICATION (
            POLICY_ID INTEGER NOT NULL,
            NAME VARCHAR2(512) NOT NULL,
            DISPLAY_NAME VARCHAR2(512) DEFAULT NULL NULL,
            TENANT_ID INTEGER NOT NULL,
            DESCRIPTION VARCHAR2(1024) DEFAULT NULL NULL,
            QUOTA_TYPE VARCHAR2(25) NOT NULL,
            QUOTA INTEGER NOT NULL,
            QUOTA_UNIT VARCHAR2(10) DEFAULT NULL NULL,
            UNIT_TIME INTEGER NOT NULL,
            TIME_UNIT VARCHAR2(25) NOT NULL,
            IS_DEPLOYED INTEGER DEFAULT 0 NOT NULL,
			CUSTOM_ATTRIBUTES BLOB DEFAULT NULL,
			      UUID VARCHAR2(256),
            PRIMARY KEY (POLICY_ID),
            CONSTRAINT AM_POLICY_APP_NAME_TENANT UNIQUE (NAME, TENANT_ID),
            UNIQUE (UUID)
)
/
CREATE TABLE AM_API_THROTTLE_POLICY (
            POLICY_ID INTEGER NOT NULL,
            NAME VARCHAR2(512) NOT NULL,
            DISPLAY_NAME VARCHAR2(512) DEFAULT NULL NULL,
            TENANT_ID INTEGER NOT NULL,
            DESCRIPTION VARCHAR2 (1024),
            DEFAULT_QUOTA_TYPE VARCHAR2(25) NOT NULL,
            DEFAULT_QUOTA INTEGER NOT NULL,
            DEFAULT_QUOTA_UNIT VARCHAR2(10) NULL,
            DEFAULT_UNIT_TIME INTEGER NOT NULL,
            DEFAULT_TIME_UNIT VARCHAR2(25) NOT NULL,
            APPLICABLE_LEVEL VARCHAR2(25) NOT NULL,
            IS_DEPLOYED INTEGER DEFAULT 0 NOT NULL,
            UUID VARCHAR2(256),
            PRIMARY KEY (POLICY_ID),
            CONSTRAINT API_POLICY_NAME_TENANT UNIQUE (NAME, TENANT_ID),
            UNIQUE (UUID)
)
/
CREATE TABLE AM_CONDITION_GROUP (
            CONDITION_GROUP_ID INTEGER NOT NULL,
            POLICY_ID INTEGER NOT NULL,
            QUOTA_TYPE VARCHAR2(25),
            QUOTA INTEGER NOT NULL,
            QUOTA_UNIT VARCHAR2(10) DEFAULT NULL NULL,
            UNIT_TIME INTEGER NOT NULL,
            TIME_UNIT VARCHAR2(25) NOT NULL,
            DESCRIPTION VARCHAR2(1024) DEFAULT NULL NULL,
            PRIMARY KEY (CONDITION_GROUP_ID),
            FOREIGN KEY (POLICY_ID) REFERENCES AM_API_THROTTLE_POLICY(POLICY_ID)
)
/
CREATE TABLE AM_QUERY_PARAMETER_CONDITION (
            QUERY_PARAMETER_ID INTEGER NOT NULL,
            CONDITION_GROUP_ID INTEGER NOT NULL,
            PARAMETER_NAME VARCHAR2(255) DEFAULT NULL,
            PARAMETER_VALUE VARCHAR2(255) DEFAULT NULL,
	    	IS_PARAM_MAPPING CHAR(1) DEFAULT 1,
            PRIMARY KEY (QUERY_PARAMETER_ID),
            FOREIGN KEY (CONDITION_GROUP_ID) REFERENCES AM_CONDITION_GROUP(CONDITION_GROUP_ID)
)
/
CREATE TABLE AM_HEADER_FIELD_CONDITION (
            HEADER_FIELD_ID INTEGER NOT NULL,
            CONDITION_GROUP_ID INTEGER NOT NULL,
            HEADER_FIELD_NAME VARCHAR2(255) DEFAULT NULL,
            HEADER_FIELD_VALUE VARCHAR2(255) DEFAULT NULL,
	    	IS_HEADER_FIELD_MAPPING CHAR(1) DEFAULT 1,
            PRIMARY KEY (HEADER_FIELD_ID),
            FOREIGN KEY (CONDITION_GROUP_ID) REFERENCES AM_CONDITION_GROUP(CONDITION_GROUP_ID)
)
/
CREATE TABLE AM_JWT_CLAIM_CONDITION (
            JWT_CLAIM_ID INTEGER NOT NULL,
            CONDITION_GROUP_ID INTEGER NOT NULL,
            CLAIM_URI VARCHAR2(512) DEFAULT NULL,
            CLAIM_ATTRIB VARCHAR2(1024) DEFAULT NULL,
	    IS_CLAIM_MAPPING CHAR(1) DEFAULT 1,
            PRIMARY KEY (JWT_CLAIM_ID),
            FOREIGN KEY (CONDITION_GROUP_ID) REFERENCES AM_CONDITION_GROUP(CONDITION_GROUP_ID)
)
/
CREATE TABLE AM_IP_CONDITION (
  AM_IP_CONDITION_ID INTEGER NOT NULL,
  STARTING_IP VARCHAR2(45) NULL,
  ENDING_IP VARCHAR2(45) NULL,
  SPECIFIC_IP VARCHAR2(45) NULL,
  WITHIN_IP_RANGE CHAR(1) DEFAULT 1,
  CONDITION_GROUP_ID INTEGER NULL,
  PRIMARY KEY (AM_IP_CONDITION_ID)
 ,  CONSTRAINT fk_AM_IP_CONDITION_1    FOREIGN KEY (CONDITION_GROUP_ID)
    REFERENCES AM_CONDITION_GROUP (CONDITION_GROUP_ID))

/
CREATE INDEX fk_AM_IP_CONDITION_1_idx ON AM_IP_CONDITION (CONDITION_GROUP_ID ASC)
/
CREATE TABLE AM_POLICY_GLOBAL (
            POLICY_ID INTEGER NOT NULL,
            NAME VARCHAR2(512) NOT NULL,
            KEY_TEMPLATE VARCHAR2(512) NOT NULL,
            TENANT_ID INTEGER NOT NULL,
            DESCRIPTION VARCHAR2(1024) DEFAULT NULL NULL,
            SIDDHI_QUERY BLOB DEFAULT NULL,
            IS_DEPLOYED INTEGER DEFAULT 0 NOT NULL,
            UUID VARCHAR2(256),
            PRIMARY KEY (POLICY_ID),
            UNIQUE (UUID)
)
/
CREATE TABLE AM_THROTTLE_TIER_PERMISSIONS (
  THROTTLE_TIER_PERMISSIONS_ID INTEGER NOT NULL,
  TIER VARCHAR2(50) NULL,
  PERMISSIONS_TYPE VARCHAR2(50) NULL,
  ROLES VARCHAR2(512) NULL,
  TENANT_ID INTEGER NULL,
  PRIMARY KEY (THROTTLE_TIER_PERMISSIONS_ID))
/
CREATE TABLE AM_BLOCK_CONDITIONS (
  CONDITION_ID INTEGER NOT NULL,
  TYPE varchar2(45) DEFAULT NULL,
  VALUE varchar2(512) DEFAULT NULL,
  ENABLED varchar2(45) DEFAULT NULL,
  DOMAIN varchar2(45) DEFAULT NULL,
  UUID VARCHAR2(256),
  PRIMARY KEY (CONDITION_ID),
  UNIQUE (UUID)
)
/
CREATE TABLE AM_CERTIFICATE_METADATA (
  TENANT_ID INTEGER NOT NULL,
  ALIAS VARCHAR2(255) NOT NULL,
  END_POINT VARCHAR2(255) NOT NULL,
  CONSTRAINT PK_ALIAS PRIMARY KEY (ALIAS)
)
/
CREATE TABLE AM_API_CLIENT_CERTIFICATE (
  TENANT_ID INTEGER NOT NULL,
  ALIAS VARCHAR2(45) NOT NULL,
  API_ID INTEGER NOT NULL,
  CERTIFICATE BLOB NOT NULL,
  REMOVED INTEGER DEFAULT 0 NOT NULL,
  TIER_NAME VARCHAR2 (512),
  FOREIGN KEY (API_ID) REFERENCES AM_API (API_ID),
  PRIMARY KEY (ALIAS, TENANT_ID, REMOVED)
)
/
CREATE TABLE AM_APPLICATION_GROUP_MAPPING (
    APPLICATION_ID INTEGER NOT NULL,
    GROUP_ID VARCHAR2(512) NOT NULL,
    TENANT VARCHAR2 (255),
    PRIMARY KEY (APPLICATION_ID,GROUP_ID,TENANT),
    FOREIGN KEY (APPLICATION_ID) REFERENCES AM_APPLICATION(APPLICATION_ID)
)
/
CREATE TABLE AM_USAGE_UPLOADED_FILES (
  TENANT_DOMAIN VARCHAR2 (255) NOT NULL,
  FILE_NAME VARCHAR2 (255) NOT NULL,
  FILE_TIMESTAMP TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FILE_PROCESSED INTEGER DEFAULT 0,
  FILE_CONTENT BLOB DEFAULT NULL,
  PRIMARY KEY (TENANT_DOMAIN, FILE_NAME, FILE_TIMESTAMP))
/
CREATE TABLE AM_API_LC_PUBLISH_EVENTS (
    ID INTEGER,
    TENANT_DOMAIN VARCHAR2 (500) NOT NULL,
    API_ID VARCHAR2 (500) NOT NULL,
    EVENT_TIME TIMESTAMP NOT NULL,
    PRIMARY KEY (ID)
)
/
CREATE TABLE AM_LABELS (
  LABEL_ID VARCHAR2(50),
  NAME VARCHAR2(255) NOT NULL,
  DESCRIPTION VARCHAR2(1024),
  TENANT_DOMAIN VARCHAR2(255),
  UNIQUE (NAME,TENANT_DOMAIN),
  PRIMARY KEY (LABEL_ID)
)
/
CREATE TABLE AM_LABEL_URLS (
  LABEL_ID VARCHAR2(50),
  ACCESS_URL VARCHAR2(255),
  PRIMARY KEY (LABEL_ID,ACCESS_URL),
  FOREIGN KEY (LABEL_ID) REFERENCES AM_LABELS(LABEL_ID)
)
/
CREATE TABLE AM_SYSTEM_APPS (
  ID INTEGER,
  NAME VARCHAR2(50) NOT NULL,
  CONSUMER_KEY VARCHAR2(512) NOT NULL,
  CONSUMER_SECRET VARCHAR2(512) NOT NULL,
  TENANT_DOMAIN VARCHAR2(255) DEFAULT 'carbon.super',
  CREATED_TIME TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  UNIQUE (NAME),
  UNIQUE (CONSUMER_KEY),
  PRIMARY KEY (ID)
)
/
CREATE TABLE AM_APPLICATION_ATTRIBUTES (
  APPLICATION_ID INTEGER,
  NAME VARCHAR2(255),
  VALUE VARCHAR2(1024),
  TENANT_ID INTEGER,
  PRIMARY KEY (APPLICATION_ID,NAME),
  FOREIGN KEY (APPLICATION_ID) REFERENCES AM_APPLICATION (APPLICATION_ID)
)
/
-- BotDATA Email table --
CREATE TABLE AM_NOTIFICATION_SUBSCRIBER (
    UUID VARCHAR2(255),
    CATEGORY VARCHAR2(255),
    NOTIFICATION_METHOD VARCHAR2(255),
    SUBSCRIBER_ADDRESS VARCHAR2(255) NOT NULL,
    PRIMARY KEY(UUID, SUBSCRIBER_ADDRESS)
)
/
CREATE TABLE AM_REVOKED_JWT (
    UUID VARCHAR(255) NOT NULL,
    SIGNATURE VARCHAR(2048) NOT NULL,
    EXPIRY_TIMESTAMP NUMBER(19) NOT NULL,
    TENANT_ID INTEGER DEFAULT -1,
    TOKEN_TYPE VARCHAR(15) DEFAULT 'DEFAULT',
    TIME_CREATED TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (UUID)
)
/
CREATE TABLE AM_API_CATEGORIES (
  UUID VARCHAR2(50),
  NAME VARCHAR2(255) NOT NULL,
  DESCRIPTION VARCHAR2(1024),
  TENANT_ID INTEGER DEFAULT -1,
  UNIQUE (NAME,TENANT_ID),
  PRIMARY KEY (UUID)
)
/
CREATE TABLE AM_USER (
    USER_ID VARCHAR(255) NOT NULL,
    USER_NAME VARCHAR(255) NOT NULL,
    PRIMARY KEY(USER_ID)
)
/
CREATE TABLE AM_KEY_MANAGER (
  UUID VARCHAR(50) NOT NULL,
  NAME VARCHAR(100) NULL,
  DISPLAY_NAME VARCHAR(100) NULL,
  DESCRIPTION VARCHAR(256) NULL,
  TYPE VARCHAR(45) NULL,
  CONFIGURATION BLOB NULL,
  ENABLED CHAR(1) DEFAULT 1,
  TENANT_DOMAIN VARCHAR(100) NULL,
  PRIMARY KEY (UUID),
  UNIQUE (NAME,TENANT_DOMAIN)
)
/

CREATE TABLE AM_GW_PUBLISHED_API_DETAILS (
  API_ID varchar(255) NOT NULL,
  TENANT_DOMAIN varchar(255),
  API_PROVIDER varchar(255),
  API_NAME varchar(255),
  API_VERSION varchar(255),
  API_TYPE varchar(50),
  PRIMARY KEY (API_ID)
)
/

CREATE TABLE AM_GW_API_ARTIFACTS (
  API_ID varchar(255) NOT NULL,
  REVISION_ID varchar(255) NOT NULL,
  ARTIFACT blob,
  TIME_STAMP TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (REVISION_ID, API_ID),
  FOREIGN KEY (API_ID) REFERENCES AM_GW_PUBLISHED_API_DETAILS(API_ID) ON UPDATE CASCADE
)
/
CREATE TABLE AM_GW_API_DEPLOYMENTS (
  API_ID VARCHAR(255) NOT NULL,
  REVISION_ID VARCHAR(255) NOT NULL,
  LABEL VARCHAR(255) NOT NULL,
  PRIMARY KEY (REVISION_ID, API_ID,LABEL),
  FOREIGN KEY (API_ID) REFERENCES AM_GW_PUBLISHED_API_DETAILS(API_ID) ON UPDATE CASCADE
)
/
CREATE TABLE AM_SCOPE (
            SCOPE_ID INTEGER NOT NULL,
            NAME VARCHAR2(255) NOT NULL,
            DISPLAY_NAME VARCHAR2(255) NOT NULL,
            DESCRIPTION VARCHAR2(512),
            TENANT_ID INTEGER DEFAULT -1 NOT NULL,
            SCOPE_TYPE VARCHAR2(255) NOT NULL,
            PRIMARY KEY (SCOPE_ID))
/
CREATE TABLE AM_SCOPE_BINDING (
            SCOPE_ID INTEGER NOT NULL,
            SCOPE_BINDING VARCHAR2(255) NOT NULL,
            BINDING_TYPE VARCHAR2(255) NOT NULL,
            FOREIGN KEY (SCOPE_ID) REFERENCES AM_SCOPE(SCOPE_ID))
/

CREATE OR REPLACE TRIGGER update_timestamp
    BEFORE INSERT OR UPDATE ON AM_GW_API_ARTIFACTS
    FOR EACH ROW
BEGIN
    :NEW.TIME_STAMP := systimestamp;
END;
/
-- End of API-MGT Tables --

--permance indexes start--
/
create index IDX_ITS_LMT on IDN_THRIFT_SESSION (LAST_MODIFIED_TIME)
/
create index IDX_IOAT_UT on IDN_OAUTH2_ACCESS_TOKEN (USER_TYPE)
/
create index IDX_AAI_CTX on AM_API (CONTEXT)
/
create index IDX_AAKM_CK on AM_APPLICATION_KEY_MAPPING (CONSUMER_KEY)
/
create index IDX_AAUM_AI on AM_API_URL_MAPPING (API_ID)
/
create index IDX_AAPROD_AI on AM_API_PRODUCT_MAPPING (API_ID)
/
create index IDX_AAUM_TT on AM_API_URL_MAPPING (THROTTLING_TIER)
/
create index IDX_AATP_DQT on AM_API_THROTTLE_POLICY (DEFAULT_QUOTA_TYPE)
/
create index IDX_ACG_QT on AM_CONDITION_GROUP (QUOTA_TYPE)
/
create index IDX_APS_QT on AM_POLICY_SUBSCRIPTION (QUOTA_TYPE)
/
create index IDX_AS_AITIAI on AM_SUBSCRIPTION (API_ID,TIER_ID,APPLICATION_ID)
/
create index IDX_APA_QT on AM_POLICY_APPLICATION (QUOTA_TYPE)
/
create index IDX_AA_AT_CB on AM_APPLICATION (APPLICATION_TIER,CREATED_BY)
/
-- Performance indexes end--
CREATE or REPLACE TRIGGER TRG_DEL_IDN_CON_APPS
  BEFORE DELETE
    on IDN_OAUTH_CONSUMER_APPS
    FOR EACH ROW
      BEGIN
	DELETE FROM IDN_OAUTH2_SCOPE_VALIDATORS IOSV WHERE IOSV.APP_ID = :OLD.ID;
	DELETE FROM IDN_OIDC_PROPERTY IOP WHERE IOP.CONSUMER_KEY = :OLD.CONSUMER_KEY;
	DELETE FROM IDN_OIDC_REQ_OBJECT_REFERENCE IOROR WHERE IOROR.CONSUMER_KEY_ID = :OLD.ID;
        DELETE FROM IDN_OAUTH2_AUTHORIZATION_CODE IO2AC WHERE IO2AC.CONSUMER_KEY_ID = :OLD.ID;
        DELETE FROM IDN_OAUTH2_ACCESS_TOKEN IO2AT WHERE IO2AT.CONSUMER_KEY_ID = :OLD.ID;
        DELETE FROM IDN_OAUTH1A_ACCESS_TOKEN IOAT WHERE IOAT.CONSUMER_KEY_ID = :OLD.ID;
        DELETE FROM IDN_OAUTH1A_REQUEST_TOKEN IORT WHERE IORT.CONSUMER_KEY_ID = :OLD.ID;
	DELETE FROM IDN_OAUTH2_DEVICE_FLOW IODF WHERE IODF.CONSUMER_KEY_ID = :OLD.ID;
	DELETE FROM IDN_OAUTH2_CIBA_AUTH_CODE IOCAC WHERE IOCAC.CONSUMER_KEY = :OLD.CONSUMER_KEY;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_IDN_OA2_ACC
  BEFORE DELETE
    on IDN_OAUTH2_ACCESS_TOKEN
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OAUTH2_ACCESS_TOKEN_SCOPE IO2ATS WHERE IO2ATS.TOKEN_ID = :OLD.TOKEN_ID;
	DELETE FROM IDN_OIDC_REQ_OBJECT_REFERENCE IOROR WHERE IOROR.TOKEN_ID = :OLD.TOKEN_ID;
	DELETE FROM IDN_OAUTH2_TOKEN_BINDING IOTB WHERE IOTB.TOKEN_ID = :OLD.TOKEN_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_DEV_FLOW
  BEFORE DELETE
    on IDN_OAUTH2_DEVICE_FLOW
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OAUTH2_DEVICE_FLOW_SCOPES IODFS WHERE IODFS.SCOPE_ID = :OLD.CODE_ID;
      END;
/

CREATE OR REPLACE TRIGGER TRG_DEL_AM_GW_PUBLISHED_API_DETAILS
  BEFORE DELETE
    on AM_GW_PUBLISHED_API_DETAILS
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_GW_API_ARTIFACTS AGA WHERE AGA.API_ID = :OLD.API_ID;
      END;
/

CREATE OR REPLACE TRIGGER TRG_DEL_IDN_OAUTH2_SCOPE
  BEFORE DELETE
    on IDN_OAUTH2_SCOPE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OAUTH2_SCOPE_BINDING IOSB WHERE IOSB.SCOPE_ID = :OLD.SCOPE_ID;
	DELETE FROM IDN_OAUTH2_RESOURCE_SCOPE IORS WHERE IORS.SCOPE_ID = :OLD.SCOPE_ID;
        DELETE FROM AM_API_SCOPES AMS WHERE AMS.SCOPE_ID = :OLD.SCOPE_ID;
	DELETE FROM IDN_OIDC_SCOPE_CLAIM_MAPPING IOSCM WHERE IOSCM.SCOPE_ID = :OLD.SCOPE_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_SP_APP
  BEFORE DELETE
    on SP_APP
    FOR EACH ROW
      BEGIN
        DELETE FROM SP_AUTH_STEP SAS WHERE SAS.APP_ID = :OLD.ID;
        DELETE FROM SP_REQ_PATH_AUTHENTICATOR SRPA WHERE SRPA.APP_ID = :OLD.ID;
        DELETE FROM SP_PROVISIONING_CONNECTOR SPC WHERE SPC.APP_ID = :OLD.ID;
        DELETE FROM SP_METADATA META WHERE META.SP_ID = :OLD.ID;
        DELETE FROM SP_INBOUND_AUTH SIA WHERE SIA.APP_ID = :OLD.ID;
        DELETE FROM SP_CLAIM_MAPPING SCM WHERE SCM.APP_ID = :OLD.ID;
        DELETE FROM SP_ROLE_MAPPING SRM WHERE SRM.APP_ID = :OLD.ID;
        DELETE FROM SP_CLAIM_DIALECT SCD WHERE SCD.APP_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_SP_AUTH_STEP
  BEFORE DELETE
    on SP_AUTH_STEP
    FOR EACH ROW
      BEGIN
        DELETE FROM SP_FEDERATED_IDP AFI WHERE AFI.ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDP
  BEFORE DELETE
    on IDP
    FOR EACH ROW
      BEGIN
        DELETE FROM IDP_ROLE IR WHERE IR.IDP_ID = :OLD.ID;
        DELETE FROM IDP_CLAIM IC WHERE IC.IDP_ID = :OLD.ID;
        DELETE FROM IDP_AUTHENTICATOR IA WHERE IA.IDP_ID = :OLD.ID;
        DELETE FROM IDP_METADATA IM WHERE IM.IDP_ID = :OLD.ID;
        DELETE FROM IDP_PROVISIONING_CONFIG IPC WHERE IPC.IDP_ID = :OLD.ID;
        DELETE FROM IDP_LOCAL_CLAIM ILC WHERE ILC.IDP_ID = :OLD.ID;
        DELETE FROM IDN_ASSOCIATED_ID IAI WHERE IAI.IDP_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDP_ROLE
  BEFORE DELETE
    on IDP_ROLE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDP_ROLE_MAPPING IRM WHERE IRM.IDP_ROLE_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDP_CLAIM
  BEFORE DELETE
    on IDP_CLAIM
    FOR EACH ROW
      BEGIN
        DELETE FROM IDP_CLAIM_MAPPING ICM WHERE ICM.IDP_CLAIM_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDP_AUTHENTICATOR
  BEFORE DELETE
    on IDP_AUTHENTICATOR
    FOR EACH ROW
      BEGIN
        DELETE FROM IDP_AUTHENTICATOR_PROPERTY IAP WHERE IAP.AUTHENTICATOR_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDP_PRO_CONFIG
  BEFORE DELETE
    on IDP_PROVISIONING_CONFIG
    FOR EACH ROW
      BEGIN
        DELETE FROM IDP_PROV_CONFIG_PROPERTY IPCP WHERE IPCP.PROVISIONING_CONFIG_ID = :OLD.ID;
        DELETE FROM IDP_PROVISIONING_ENTITY IPE WHERE IPE.PROVISIONING_CONFIG_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_WF_WORKFLOW
  BEFORE DELETE
    on WF_WORKFLOW
    FOR EACH ROW
      BEGIN
        DELETE FROM WF_WORKFLOW_ASSOCIATION WWA WHERE WWA.WORKFLOW_ID = :OLD.ID;
        DELETE FROM WF_WORKFLOW_CONFIG_PARAM WWCP WHERE WWCP.WORKFLOW_ID = :OLD.ID;
        DELETE FROM WF_WORKFLOW_REQUEST_RELATION WWRR WHERE WWRR.WORKFLOW_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_CLAIM_DIALECT
  BEFORE DELETE
    on IDN_CLAIM_DIALECT
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_CLAIM ICL WHERE ICL.DIALECT_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_CLAIM
  BEFORE DELETE
    on IDN_CLAIM
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_CLAIM_PROPERTY ICPR WHERE ICPR.LOCAL_CLAIM_ID = :OLD.ID;
        DELETE FROM IDN_CLAIM_MAPPED_ATTRIBUTE ICMA WHERE ICMA.LOCAL_CLAIM_ID = :OLD.ID;
        DELETE FROM IDN_CLAIM_MAPPING ICM01 WHERE ICM01.EXT_CLAIM_ID = :OLD.ID;
        DELETE FROM IDN_CLAIM_MAPPING ICM02 WHERE ICM02.MAPPED_LOCAL_CLAIM_ID = :OLD.ID;
	DELETE FROM IDN_OIDC_SCOPE_CLAIM_MAPPING IOSCM WHERE IOSCM.EXTERNAL_CLAIM_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_AUTH_CODE
  BEFORE DELETE
    on IDN_OAUTH2_AUTHORIZATION_CODE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OIDC_REQ_OBJECT_REFERENCE IOROR WHERE IOROR.CODE_ID = :OLD.CODE_ID;
        DELETE FROM IDN_OAUTH2_AUTHZ_CODE_SCOPE IOACS WHERE IOACS.CODE_ID = :OLD.CODE_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_REQ_OBJ_REF
  BEFORE DELETE
    on IDN_OIDC_REQ_OBJECT_REFERENCE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OIDC_REQ_OBJECT_CLAIMS IOROC WHERE IOROC.REQ_OBJECT_ID = :OLD.ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_IDN_OIDC_OBJ_CLAIMS
  BEFORE DELETE
    on IDN_OIDC_REQ_OBJECT_CLAIMS
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OIDC_REQ_OBJ_CLAIM_VALUES IOROCV WHERE IOROCV.REQ_OBJECT_CLAIMS_ID = :OLD.ID;
      END;
/

CREATE or REPLACE TRIGGER TRG_DEL_IDN_CIBA_AUTH
  BEFORE DELETE
    on IDN_OAUTH2_CIBA_AUTH_CODE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_OAUTH2_CIBA_AUTH_CODE IOCAC WHERE IOCAC.AUTH_CODE_KEY = :OLD.AUTH_CODE_KEY;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_IDN_PER_TICKET
  BEFORE DELETE
    on IDN_UMA_PERMISSION_TICKET
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_UMA_PT_RESOURCE IUPR WHERE IUPR.PT_ID = :OLD.ID;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_IDN_UMA_RESOURCE
  BEFORE DELETE
    on IDN_UMA_RESOURCE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_UMA_PT_RESOURCE IUPR WHERE IUPR.PT_RESOURCE_ID = :OLD.ID;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_IDN_UMA_PT_RESOURCE
  BEFORE DELETE
    on IDN_UMA_PT_RESOURCE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_UMA_PT_RESOURCE_SCOPE IUPRS WHERE IUPRS.PT_RESOURCE_ID = :OLD.ID;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_IDN_UMA_RESOURCE_SCOPE
  BEFORE DELETE
    on IDN_UMA_RESOURCE_SCOPE
    FOR EACH ROW
      BEGIN
        DELETE FROM IDN_UMA_PT_RESOURCE_SCOPE IUPRS WHERE IUPRS.PT_SCOPE_ID = :OLD.ID;
      END;
/

CREATE OR REPLACE TRIGGER TRG_DEL_AM_SUBSCRIBER
  BEFORE DELETE
    on AM_SUBSCRIBER
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_APPLICATION AMA WHERE AMA.SUBSCRIBER_ID = :OLD.SUBSCRIBER_ID;
        DELETE FROM AM_API_RATINGS AMAR WHERE AMAR.SUBSCRIBER_ID = :OLD.SUBSCRIBER_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_AM_API
  BEFORE DELETE
    on AM_API
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_SUBSCRIPTION AMSU WHERE AMSU.API_ID = :OLD.API_ID;
        DELETE FROM AM_API_LC_EVENT AMLE WHERE AMLE.API_ID = :OLD.API_ID;
        DELETE FROM AM_API_COMMENTS AMAC WHERE AMAC.API_ID = :OLD.API_ID;
        DELETE FROM AM_API_RATINGS AMAR WHERE AMAR.API_ID = :OLD.API_ID;
        DELETE FROM AM_EXTERNAL_STORES AMES WHERE AMES.API_ID = :OLD.API_ID;
        DELETE FROM AM_API_SCOPES AMS WHERE AMS.API_ID = :OLD.API_ID;
	DELETE FROM AM_API_PRODUCT_MAPPING AAPM WHERE AAPM.API_ID = :OLD.API_ID;
	DELETE FROM AM_API_CLIENT_CERTIFICATE AACC WHERE AACC.API_ID = :OLD.API_ID;
	DELETE FROM AM_SECURITY_AUDIT_UUID_MAPPING ASAUM WHERE ASAUM.API_ID = :OLD.API_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_AM_API_URL_MAPPING
  BEFORE DELETE
    on AM_API_URL_MAPPING
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_API_PRODUCT_MAPPING AAPM WHERE AAPM.URL_MAPPING_ID = :OLD.URL_MAPPING_ID;
      END;
/
CREATE or REPLACE TRIGGER TRG_DEL_AM_APPLICATION
  BEFORE DELETE
    on AM_APPLICATION
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_SUBSCRIPTION AMSU WHERE AMSU.APPLICATION_ID = :OLD.APPLICATION_ID;
        DELETE FROM AM_APPLICATION_KEY_MAPPING AMKM WHERE AMKM.APPLICATION_ID = :OLD.APPLICATION_ID;
        DELETE FROM AM_APPLICATION_REGISTRATION AMAR WHERE AMAR.APP_ID = :OLD.APPLICATION_ID;
        DELETE FROM AM_APPLICATION_GROUP_MAPPING AMGM WHERE AMGM.APPLICATION_ID = :OLD.APPLICATION_ID;
	DELETE FROM AM_APPLICATION_ATTRIBUTES AAA WHERE AAA.APPLICATION_ID = :OLD.APPLICATION_ID;
      END;
/

CREATE OR REPLACE TRIGGER TRG_DEL_AM_API_THROTTLE_POLICY
  BEFORE DELETE
    on AM_API_THROTTLE_POLICY
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_CONDITION_GROUP AMCG WHERE AMCG.POLICY_ID = :OLD.POLICY_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_AM_CONDITION_GROUP
  BEFORE DELETE
    on AM_CONDITION_GROUP
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_QUERY_PARAMETER_CONDITION AMQPC WHERE AMQPC.CONDITION_GROUP_ID = :OLD.CONDITION_GROUP_ID;
        DELETE FROM AM_HEADER_FIELD_CONDITION AMHFC WHERE AMHFC.CONDITION_GROUP_ID = :OLD.CONDITION_GROUP_ID;
        DELETE FROM AM_JWT_CLAIM_CONDITION AMJCC WHERE AMJCC.CONDITION_GROUP_ID = :OLD.CONDITION_GROUP_ID;
        DELETE FROM AM_IP_CONDITION AMIC WHERE AMIC.CONDITION_GROUP_ID = :OLD.CONDITION_GROUP_ID;
      END;
/
CREATE OR REPLACE TRIGGER TRG_DEL_AM_LABELS
  BEFORE DELETE
    on AM_LABELS
    FOR EACH ROW
      BEGIN
        DELETE FROM AM_LABEL_URLS ALU WHERE ALU.LABEL_ID = :OLD.LABEL_ID;
      END;
/
