/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.carbon.directory.server.manager.internal;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import org.apache.axiom.om.util.Base64;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.carbon.directory.server.manager.DirectoryServerManagerException;
import org.wso2.carbon.directory.server.manager.common.ServerPrinciple;
import org.wso2.carbon.directory.server.manager.internal.LDAPServerStoreManagerUtil;
import org.wso2.carbon.user.api.RealmConfiguration;
import org.wso2.carbon.user.core.UserStoreException;
import org.wso2.carbon.user.core.ldap.LDAPConnectionContext;
import org.wso2.carbon.user.core.util.JNDIUtil;

public class LDAPServerStoreManager {
    private static final Log log = LogFactory.getLog(LDAPServerStoreManager.class);
    private LDAPConnectionContext connectionSource;
    private RealmConfiguration realmConfiguration;

    public LDAPServerStoreManager(RealmConfiguration realmConfig) {
        this.realmConfiguration = realmConfig;
        try {
            this.connectionSource = new LDAPConnectionContext(realmConfig);
        }
        catch (UserStoreException e) {
            log.error((Object)"Error occurred while instantiating LDAPConnectionContext", (Throwable)e);
        }
    }

    protected boolean isServerNameValid(String serverName) {
        String serviceNamePolicyRegEx = this.realmConfiguration.getUserStoreProperty("ServiceNameJavaRegEx");
        if (serviceNamePolicyRegEx == null) {
            serviceNamePolicyRegEx = "[a-zA-Z\\d]{2,10}/[a-zA-Z]{2,30}";
        }
        log.info((Object)("Using service name format - " + serviceNamePolicyRegEx));
        return this.isFormatCorrect(serviceNamePolicyRegEx, serverName);
    }

    protected boolean isPasswordValid(String password) {
        String regularExpression = this.realmConfiguration.getUserStoreProperty("ServicePasswordJavaRegEx");
        if (regularExpression == null) {
            regularExpression = "[\\S]{5,30}";
        }
        log.info((Object)("Using service password format - " + regularExpression));
        return StringUtils.isNotEmpty((String)password) && this.isFormatCorrect(regularExpression, password);
    }

    private boolean isFormatCorrect(String regularExpression, String attribute) {
        Pattern p = Pattern.compile(regularExpression);
        Matcher m = p.matcher(attribute);
        return m.matches();
    }

    public String getServiceName(String serverName) throws DirectoryServerManagerException {
        String[] components = serverName.split("/");
        if (components.length != 2) {
            throw new DirectoryServerManagerException("Invalid server name provided. Could not retrieve service component.");
        }
        if (this.isExistingServiceUid(components[0])) {
            return this.getUniqueServiceUid(serverName);
        }
        return components[0];
    }

    protected String getUniqueServiceUid(String serviceName) {
        String[] domainParts;
        String[] parts = serviceName.split("/");
        if (parts.length == 1) {
            return parts[0];
        }
        StringBuilder uniqueId = new StringBuilder(parts[0]);
        for (String domainPart : domainParts = parts[1].split("\\.")) {
            uniqueId.append("-").append(domainPart);
        }
        return uniqueId.toString();
    }

    protected String getServerPrincipleExcludeString() {
        return this.getServiceFilteringExpression(true);
    }

    protected String getServerPrincipleIncludeString() {
        return this.getServiceFilteringExpression(false);
    }

    private String getServiceFilteringExpression(boolean excludeServer) {
        if (excludeServer) {
            return "(!(sn=Service))";
        }
        return "(sn=Service)";
    }

    public boolean isExistingServiceUid(String uid) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            log.error((Object)"Unable to retrieve directory context.", (Throwable)e);
            throw new DirectoryServerManagerException("Unable to retrieve directory context.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String filter = "(&(uid=" + uid + ")" + this.getServerPrincipleIncludeString() + ")";
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"uid"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, filter, searchControls);
            boolean bl = namingEnumeration.hasMore();
            return bl;
        }
        catch (NamingException e) {
            log.error((Object)("Unable to check whether service exists in directory server. UID - " + uid), (Throwable)e);
            throw new DirectoryServerManagerException("Can not access the directory service", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    public boolean isExistingServicePrinciple(String servicePrinciple) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            log.error((Object)"Unable to retrieve directory context.", (Throwable)e);
            throw new DirectoryServerManagerException("Unable to retrieve directory context.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String filter = this.getServicePrincipleFilter(servicePrinciple);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"uid"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, filter, searchControls);
            boolean bl = namingEnumeration.hasMore();
            return bl;
        }
        catch (NamingException e) {
            String message = "Unable to search entry with search base " + searchBase + ", filter -" + filter;
            log.error((Object)message, (Throwable)e);
            throw new DirectoryServerManagerException("Can not access the directory service", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    public void addServicePrinciple(String serverName, String serverDescription, Object credentials) throws DirectoryServerManagerException {
        DirContext dirContext;
        if (!(credentials instanceof String)) {
            throw new DirectoryServerManagerException("Invalid credentials provided");
        }
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("An error occurred while retrieving LDAP connection context.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        try {
            dirContext = (DirContext)dirContext.lookup(searchBase);
            BasicAttributes basicAttributes = new BasicAttributes(true);
            String serverUid = this.getServiceName(serverName);
            this.constructBasicAttributes(basicAttributes, serverUid, serverName, credentials, serverDescription, "Service");
            dirContext.bind("uid=" + serverUid, null, (Attributes)basicAttributes);
        }
        catch (NamingException e) {
            String message = "Can not access the directory context or user already exists in the system";
            log.error((Object)message, (Throwable)e);
            throw new DirectoryServerManagerException(message, e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    private void constructBasicAttributes(BasicAttributes basicAttributes, String id, String principleName, Object credential, String commonName, String surName) throws DirectoryServerManagerException {
        BasicAttribute objectClass = new BasicAttribute("objectClass");
        objectClass.add("inetOrgPerson");
        objectClass.add("organizationalPerson");
        objectClass.add("person");
        objectClass.add("top");
        objectClass.add("krb5principal");
        objectClass.add("krb5kdcentry");
        objectClass.add("subschema");
        basicAttributes.put(objectClass);
        BasicAttribute uid = new BasicAttribute("uid");
        uid.add(id);
        basicAttributes.put(uid);
        String principal = this.getFullyQualifiedPrincipalName(principleName);
        BasicAttribute principalAttribute = new BasicAttribute("krb5PrincipalName");
        principalAttribute.add(principal);
        basicAttributes.put(principalAttribute);
        BasicAttribute versionNumberAttribute = new BasicAttribute("krb5KeyVersionNumber");
        versionNumberAttribute.add("0");
        basicAttributes.put(versionNumberAttribute);
        BasicAttribute userPassword = new BasicAttribute("userPassword");
        String password = this.getPasswordToStore((String)credential, "PlainText");
        userPassword.add(password.getBytes());
        basicAttributes.put(userPassword);
        if (commonName == null || commonName.isEmpty()) {
            commonName = principleName + " Service";
        }
        BasicAttribute cn = new BasicAttribute("cn");
        cn.add(commonName);
        basicAttributes.put(cn);
        BasicAttribute sn = new BasicAttribute("sn");
        sn.add(surName);
        basicAttributes.put(sn);
    }

    public ServerPrinciple[] listServicePrinciples(String filter) throws DirectoryServerManagerException {
        Object[] serverNames = null;
        int maxItemLimit = Integer.parseInt(this.realmConfiguration.getUserStoreProperty("MaxUserNameListLength"));
        SearchControls searchCtls = new SearchControls();
        searchCtls.setSearchScope(2);
        searchCtls.setCountLimit(maxItemLimit);
        if (filter.contains("?") || filter.contains("**")) {
            log.error((Object)("Invalid search character " + filter));
            throw new DirectoryServerManagerException("Invalid character sequence entered for service principle search. Please enter valid sequence.");
        }
        StringBuilder searchFilter = new StringBuilder(this.realmConfiguration.getUserStoreProperty("UserNameListFilter"));
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        StringBuilder buff = new StringBuilder();
        buff.append("(&").append((CharSequence)searchFilter).append("(").append("krb5PrincipalName").append("=").append(filter).append(")").append(this.getServerPrincipleIncludeString()).append(")");
        String[] returnedAtts = new String[]{"krb5PrincipalName", "cn"};
        searchCtls.setReturningAttributes(returnedAtts);
        DirContext dirContext = null;
        try {
            dirContext = this.connectionSource.getContext();
            NamingEnumeration<SearchResult> answer = dirContext.search(searchBase, buff.toString(), searchCtls);
            ArrayList<ServerPrinciple> list = new ArrayList<ServerPrinciple>();
            int i = 0;
            while (answer.hasMoreElements() && i < maxItemLimit) {
                String serverPrincipleFullName;
                SearchResult sr = answer.next();
                if (sr.getAttributes() == null) continue;
                Attribute serverNameAttribute = sr.getAttributes().get("krb5PrincipalName");
                Attribute serverDescription = sr.getAttributes().get("cn");
                if (serverNameAttribute == null || (serverPrincipleFullName = (String)serverNameAttribute.get()).toLowerCase(Locale.ENGLISH).contains("krbtgt")) continue;
                String serviceName = serverPrincipleFullName.contains("@") ? serverPrincipleFullName.split("@")[0] : serverPrincipleFullName;
                ServerPrinciple principle = serverDescription != null ? new ServerPrinciple(serviceName, (String)serverDescription.get()) : new ServerPrinciple(serviceName);
                list.add(principle);
                ++i;
            }
            serverNames = list.toArray(new ServerPrinciple[list.size()]);
            Arrays.sort(serverNames);
        }
        catch (NamingException e) {
            log.error((Object)e.getMessage(), (Throwable)e);
            throw new DirectoryServerManagerException("Unable to list service principles.", e);
        }
        catch (UserStoreException e) {
            log.error((Object)"Unable to retrieve LDAP connection context.", (Throwable)e);
            throw new DirectoryServerManagerException("Unable to list service principles.", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
        return serverNames;
    }

    private String getFullyQualifiedPrincipalName(String principleName) {
        String defaultRealmName = this.getRealmName();
        return principleName.toLowerCase(Locale.US) + "@" + defaultRealmName.toUpperCase(Locale.ENGLISH);
    }

    private String getPasswordToStore(String password, String passwordHashMethod) throws DirectoryServerManagerException {
        String passwordToStore = password;
        if (passwordHashMethod != null) {
            try {
                if (passwordHashMethod.equals("PlainText")) {
                    return passwordToStore;
                }
                MessageDigest messageDigest = MessageDigest.getInstance(passwordHashMethod);
                byte[] digestValue = messageDigest.digest(password.getBytes(StandardCharsets.UTF_8));
                passwordToStore = "{" + passwordHashMethod + "}" + Base64.encode((byte[])digestValue);
            }
            catch (NoSuchAlgorithmException e) {
                throw new DirectoryServerManagerException("Invalid hashMethod", e);
            }
        }
        return passwordToStore;
    }

    private String getServicePrincipleFilter(String servicePrincipleName) {
        String serverNameInRealm = this.getFullyQualifiedPrincipalName(LDAPServerStoreManagerUtil.escapeSpecialCharactersForFilter(servicePrincipleName));
        return "(&(krb5PrincipalName=" + serverNameInRealm + ")" + this.getServerPrincipleIncludeString() + ")";
    }

    private Attribute getChangePasswordAttribute(Attribute oldPasswordAttribute, Object oldCredential, Object newPassword) throws DirectoryServerManagerException {
        String passwordHashMethod = null;
        if (oldCredential != null) {
            try {
                NamingEnumeration<?> passwords = oldPasswordAttribute.getAll();
                if (passwords.hasMore()) {
                    byte[] byteArray = (byte[])passwords.next();
                    String password = new String(byteArray, StandardCharsets.UTF_8);
                    if (password.startsWith("{")) {
                        passwordHashMethod = password.substring(password.indexOf("{") + 1, password.indexOf("}"));
                    }
                    if (!password.equals(this.getPasswordToStore((String)oldCredential, passwordHashMethod))) {
                        throw new DirectoryServerManagerException("Old password does not match");
                    }
                }
            }
            catch (NamingException e) {
                log.error((Object)"Unable to retrieve old password details.", (Throwable)e);
                throw new DirectoryServerManagerException("Could not find old password details");
            }
        }
        BasicAttribute passwordAttribute = new BasicAttribute("userPassword");
        passwordAttribute.add(this.getPasswordToStore((String)newPassword, passwordHashMethod));
        return passwordAttribute;
    }

    public void updateServicePrinciplePassword(String serverName, Object oldCredential, Object newCredentials) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("Unable to retrieve directory connection.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String searchFilter = this.getServicePrincipleFilter(serverName);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"userPassword"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, searchFilter, searchControls);
            while (namingEnumeration.hasMore()) {
                BasicAttributes basicAttributes = new BasicAttributes(true);
                SearchResult searchResult = namingEnumeration.next();
                Attributes attributes = searchResult.getAttributes();
                Attribute userPassword = attributes.get("userPassword");
                Attribute newPasswordAttribute = this.getChangePasswordAttribute(userPassword, oldCredential, newCredentials);
                basicAttributes.put(newPasswordAttribute);
                String dnName = searchResult.getName();
                dirContext = (DirContext)dirContext.lookup(searchBase);
                dirContext.modifyAttributes(dnName, 2, (Attributes)basicAttributes);
            }
        }
        catch (NamingException e) {
            log.error((Object)("Unable to update server principle password details. Server name - " + serverName));
            throw new DirectoryServerManagerException("Can not access the directory service", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    public ServerPrinciple getServicePrinciple(String serverName) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("Unable to retrieve directory connection.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String searchFilter = this.getServicePrincipleFilter(serverName);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"userPassword", "cn"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, searchFilter, searchControls);
            if (namingEnumeration.hasMore()) {
                BasicAttributes basicAttributes = new BasicAttributes(true);
                SearchResult searchResult = namingEnumeration.next();
                Attributes attributes = searchResult.getAttributes();
                String userPassword = (String)attributes.get("userPassword").get();
                String description = (String)attributes.get("cn").get();
                ServerPrinciple serverPrinciple = new ServerPrinciple(serverName, description, userPassword);
                return serverPrinciple;
            }
            ServerPrinciple serverPrinciple = null;
            return serverPrinciple;
        }
        catch (NamingException e) {
            throw new DirectoryServerManagerException("Can not access the directory service", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    public boolean isValidPassword(String serverName, Object existingCredentials) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("Unable to retrieve directory connection.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String searchFilter = this.getServicePrincipleFilter(serverName);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"userPassword"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, searchFilter, searchControls);
            while (namingEnumeration.hasMore()) {
                SearchResult searchResult = namingEnumeration.next();
                Attributes attributes = searchResult.getAttributes();
                Attribute userPassword = attributes.get("userPassword");
                NamingEnumeration<?> passwords = userPassword.getAll();
                String passwordHashMethod = null;
                if (!passwords.hasMore()) continue;
                byte[] byteArray = (byte[])passwords.next();
                String password = new String(byteArray, StandardCharsets.UTF_8);
                if (password.startsWith("{")) {
                    passwordHashMethod = password.substring(password.indexOf("{") + 1, password.indexOf("}"));
                }
                boolean bl = password.equals(this.getPasswordToStore((String)existingCredentials, passwordHashMethod));
                return bl;
            }
        }
        catch (NamingException e) {
            log.error((Object)"Failed, validating password. Can not access the directory service", (Throwable)e);
            throw new DirectoryServerManagerException("Failed, validating password. Can not access the directory service", e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
        return false;
    }

    private String lookupUserId(String serverName) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("Unable to retrieve directory connection.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String searchFilter = this.getServicePrincipleFilter(serverName);
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        searchControls.setReturningAttributes(new String[]{"uid"});
        try {
            NamingEnumeration<SearchResult> namingEnumeration = dirContext.search(searchBase, searchFilter, searchControls);
            if (namingEnumeration.hasMore()) {
                SearchResult searchResult = namingEnumeration.next();
                Attributes attributes = searchResult.getAttributes();
                Attribute userId = attributes.get("uid");
                String string = (String)userId.get();
                return string;
            }
            String string = null;
            return string;
        }
        catch (NamingException e) {
            log.error((Object)("Could not find user id for given server " + serverName), (Throwable)e);
            throw new DirectoryServerManagerException("Could not find user id for given server " + serverName, e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    public void deleteServicePrinciple(String serverName) throws DirectoryServerManagerException {
        DirContext dirContext;
        try {
            dirContext = this.connectionSource.getContext();
        }
        catch (UserStoreException e) {
            throw new DirectoryServerManagerException("Unable to retrieve directory connection.", e);
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String userId = this.lookupUserId(serverName);
        if (userId == null) {
            throw new DirectoryServerManagerException("Could not find user id for given server principle " + serverName);
        }
        try {
            dirContext = (DirContext)dirContext.lookup(searchBase);
            dirContext.unbind("uid=" + userId);
        }
        catch (NamingException e) {
            log.error((Object)("Could not remove service principle " + serverName), (Throwable)e);
            throw new DirectoryServerManagerException("Could not remove service principle " + serverName, e);
        }
        finally {
            try {
                JNDIUtil.closeContext((DirContext)dirContext);
            }
            catch (UserStoreException e) {
                log.error((Object)"Unable to close directory context.", (Throwable)e);
            }
        }
    }

    private String getRealmName() {
        String defaultRealmName = this.realmConfiguration.getUserStoreProperty("defaultRealmName");
        if (defaultRealmName != null) {
            return defaultRealmName;
        }
        String searchBase = this.realmConfiguration.getUserStoreProperty("UserSearchBase");
        String[] domainComponents = searchBase.split("dc=");
        StringBuilder builder = new StringBuilder();
        for (String dc : domainComponents) {
            if (dc.contains("=")) continue;
            String trimmedDc = dc.trim();
            if (trimmedDc.endsWith(",")) {
                builder.append(trimmedDc.replace(',', '.'));
                continue;
            }
            builder.append(trimmedDc);
        }
        return builder.toString().toUpperCase(Locale.ENGLISH);
    }
}

