/*
 * Decompiled with CFR 0.152.
 */
package org.apache.synapse.transport.certificatevalidation.crl;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.synapse.transport.certificatevalidation.CertificateVerificationException;
import org.apache.synapse.transport.certificatevalidation.RevocationStatus;
import org.apache.synapse.transport.certificatevalidation.RevocationVerifier;
import org.apache.synapse.transport.certificatevalidation.crl.CRLCache;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.DERIA5String;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;

public class CRLVerifier
implements RevocationVerifier {
    private CRLCache cache;
    private static final Log log = LogFactory.getLog(CRLVerifier.class);

    public CRLVerifier(CRLCache cache) {
        this.cache = cache;
    }

    @Override
    public RevocationStatus checkRevocationStatus(X509Certificate peerCert, X509Certificate issuerCert) throws CertificateVerificationException {
        List<String> list = this.getCrlDistributionPoints(peerCert);
        for (String crlUrl : list) {
            X509CRL x509CRL;
            log.info((Object)("Trying to get CRL for URL: " + crlUrl));
            if (this.cache != null && (x509CRL = this.cache.getCacheValue(crlUrl)) != null) {
                RevocationStatus status = this.getRevocationStatus(x509CRL, peerCert);
                log.info((Object)"CRL taken from cache....");
                return status;
            }
            try {
                x509CRL = this.downloadCRLFromWeb(crlUrl);
                if (x509CRL == null) continue;
                if (this.cache != null) {
                    this.cache.setCacheValue(crlUrl, x509CRL);
                }
                return this.getRevocationStatus(x509CRL, peerCert);
            }
            catch (Exception e) {
                log.info((Object)"Either url is bad or cant build X509CRL. So check with the next url in the list.", (Throwable)e);
            }
        }
        throw new CertificateVerificationException("Cannot check revocation status with the certificate");
    }

    private RevocationStatus getRevocationStatus(X509CRL x509CRL, X509Certificate peerCert) {
        if (x509CRL.isRevoked(peerCert)) {
            return RevocationStatus.REVOKED;
        }
        return RevocationStatus.GOOD;
    }

    protected X509CRL downloadCRLFromWeb(String crlURL) throws IOException, CertificateVerificationException {
        try (InputStream crlStream = null;){
            URL url = new URL(crlURL);
            crlStream = url.openStream();
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            X509CRL x509CRL = (X509CRL)cf.generateCRL(crlStream);
            return x509CRL;
        }
    }

    private List<String> getCrlDistributionPoints(X509Certificate cert) throws CertificateVerificationException {
        CRLDistPoint distPoint;
        byte[] crlDPExtensionValue = cert.getExtensionValue(Extension.cRLDistributionPoints.getId());
        if (crlDPExtensionValue == null) {
            throw new CertificateVerificationException("Certificate doesn't have CRL Distribution points");
        }
        ASN1InputStream asn1In = new ASN1InputStream(crlDPExtensionValue);
        try {
            DEROctetString crlDEROctetString = (DEROctetString)asn1In.readObject();
            ASN1InputStream asn1InOctets = new ASN1InputStream(crlDEROctetString.getOctets());
            ASN1Primitive crlDERObject = asn1InOctets.readObject();
            distPoint = CRLDistPoint.getInstance((Object)crlDERObject);
        }
        catch (IOException e) {
            throw new CertificateVerificationException("Cannot read certificate to get CRL urls", e);
        }
        ArrayList<String> crlUrls = new ArrayList<String>();
        for (DistributionPoint dp : distPoint.getDistributionPoints()) {
            GeneralName[] genNames;
            DistributionPointName dpn = dp.getDistributionPoint();
            if (dpn == null || dpn.getType() != 0) continue;
            for (GeneralName genName : genNames = GeneralNames.getInstance((Object)dpn.getName()).getNames()) {
                if (genName.getTagNo() != 6) continue;
                String url = DERIA5String.getInstance((Object)genName.getName()).getString().trim();
                crlUrls.add(url);
            }
        }
        if (crlUrls.isEmpty()) {
            throw new CertificateVerificationException("Cant get CRL urls from certificate");
        }
        return crlUrls;
    }
}

