/*
 * Decompiled with CFR 0.152.
 */
package org.wso2.andes.server.cluster;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.wso2.andes.configuration.AndesConfigurationManager;
import org.wso2.andes.configuration.enums.AndesConfiguration;
import org.wso2.andes.kernel.AndesContext;
import org.wso2.andes.kernel.AndesContextStore;
import org.wso2.andes.kernel.AndesException;
import org.wso2.andes.server.cluster.CoordinationConfigurableClusterAgent;
import org.wso2.andes.server.cluster.CoordinationStrategy;
import org.wso2.andes.server.cluster.NodeDetail;
import org.wso2.andes.server.cluster.NodeHeartBeatData;
import org.wso2.andes.server.cluster.coordination.rdbms.MembershipEventType;
import org.wso2.andes.server.cluster.coordination.rdbms.RDBMSMembershipEventingEngine;
import org.wso2.andes.server.cluster.coordination.rdbms.RDBMSMembershipListener;

class RDBMSCoordinationStrategy
implements CoordinationStrategy,
RDBMSMembershipListener {
    private Log logger = LogFactory.getLog(RDBMSCoordinationStrategy.class);
    private RDBMSMembershipEventingEngine membershipEventingEngine;
    private InetSocketAddress thriftAddress;
    private int coordinatorEntryCreationWaitTime;
    private InetSocketAddress hazelcastAddress;
    private final int heartBeatInterval;
    private final int heartbeatMaxAge;
    private CoordinationConfigurableClusterAgent configurableClusterAgent;
    private CoordinatorElectionTask coordinatorElectionTask;
    private NodeState currentNodeState;
    private AndesContextStore contextStore;
    private String localNodeId;
    private final ExecutorService threadExecutor;
    private final ScheduledExecutorService scheduledExecutorService;

    RDBMSCoordinationStrategy() {
        ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("RDBMSCoordinationStrategy-%d").build();
        this.threadExecutor = Executors.newSingleThreadExecutor(namedThreadFactory);
        this.scheduledExecutorService = Executors.newSingleThreadScheduledExecutor();
        this.heartBeatInterval = (Integer)AndesConfigurationManager.readValue(AndesConfiguration.RDBMS_BASED_COORDINATION_HEARTBEAT_INTERVAL);
        this.coordinatorEntryCreationWaitTime = (Integer)AndesConfigurationManager.readValue(AndesConfiguration.RDBMS_BASED_COORDINATOR_ENTRY_CREATION_WAIT_TIME);
        this.heartbeatMaxAge = this.heartBeatInterval * 2;
        if (this.heartBeatInterval <= this.coordinatorEntryCreationWaitTime) {
            throw new RuntimeException("Configuration error. " + AndesConfiguration.RDBMS_BASED_COORDINATION_HEARTBEAT_INTERVAL + " * 2 should be greater than " + AndesConfiguration.RDBMS_BASED_COORDINATOR_ENTRY_CREATION_WAIT_TIME);
        }
    }

    @Override
    public void memberAdded(String nodeID) {
        this.configurableClusterAgent.memberAdded(nodeID);
    }

    @Override
    public void memberRemoved(String nodeID) {
        try {
            this.configurableClusterAgent.memberRemoved(nodeID);
        }
        catch (AndesException e) {
            this.logger.error((Object)("Error while handling node removal, " + nodeID), (Throwable)e);
        }
    }

    @Override
    public void coordinatorChanged(String coordinator) {
    }

    @Override
    public void start(CoordinationConfigurableClusterAgent configurableClusterAgent, String nodeId, InetSocketAddress thriftAddress, InetSocketAddress hazelcastAddress) {
        this.localNodeId = nodeId;
        this.thriftAddress = thriftAddress;
        this.hazelcastAddress = hazelcastAddress;
        this.contextStore = AndesContext.getInstance().getAndesContextStore();
        this.configurableClusterAgent = configurableClusterAgent;
        this.membershipEventingEngine = new RDBMSMembershipEventingEngine();
        this.membershipEventingEngine.start(nodeId);
        this.membershipEventingEngine.addEventListener(this);
        this.currentNodeState = NodeState.ELECTION;
        try {
            this.contextStore.clearMembershipEvents(nodeId);
            this.contextStore.removeNodeHeartbeat(nodeId);
        }
        catch (AndesException e) {
            this.logger.warn((Object)("Error while clearing old membership events for local node (" + nodeId + ")"), (Throwable)e);
        }
        this.coordinatorElectionTask = new CoordinatorElectionTask();
        this.threadExecutor.execute(this.coordinatorElectionTask);
        int timeout = 500;
        int waitTime = 0;
        int maxWaitTime = this.heartbeatMaxAge * 5;
        while (this.currentNodeState == NodeState.ELECTION) {
            try {
                TimeUnit.MILLISECONDS.sleep(timeout);
                if ((waitTime += timeout) != maxWaitTime) continue;
                throw new RuntimeException("Node is stuck in the ELECTION state for " + waitTime + " milliseconds.");
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("An error occurred while waiting to get current node state.", e);
            }
        }
    }

    @Override
    public boolean isCoordinator() {
        return this.currentNodeState == NodeState.COORDINATOR;
    }

    @Override
    public InetSocketAddress getThriftAddressOfCoordinator() {
        InetSocketAddress coordinatorThriftAddress = null;
        try {
            coordinatorThriftAddress = this.contextStore.getCoordinatorThriftAddress();
        }
        catch (AndesException e) {
            this.logger.error((Object)"Error occurred while reading coordinator thrift address", (Throwable)e);
        }
        return coordinatorThriftAddress;
    }

    @Override
    public List<String> getAllNodeIdentifiers() throws AndesException {
        List<NodeHeartBeatData> allNodeInformation = this.contextStore.getAllHeartBeatData();
        return this.getNodeIds(allNodeInformation);
    }

    @Override
    public List<NodeDetail> getAllNodeDetails() throws AndesException {
        ArrayList<NodeDetail> nodeDetails = new ArrayList<NodeDetail>();
        List<NodeHeartBeatData> allHeartBeatData = this.contextStore.getAllHeartBeatData();
        String coordinatorNodeId = this.contextStore.getCoordinatorNodeId();
        for (NodeHeartBeatData nodeHeartBeatData : allHeartBeatData) {
            boolean isCoordinatorNode = coordinatorNodeId.equals(nodeHeartBeatData.getNodeId());
            nodeDetails.add(new NodeDetail(nodeHeartBeatData.getNodeId(), nodeHeartBeatData.getClusterAgentAddress(), isCoordinatorNode));
        }
        return nodeDetails;
    }

    @Override
    public void stop() {
        if (this.isCoordinator()) {
            try {
                this.contextStore.removeCoordinator();
            }
            catch (AndesException e) {
                this.logger.error((Object)"Error occurred while removing coordinator when shutting down", (Throwable)e);
            }
        }
        this.membershipEventingEngine.stop();
        this.coordinatorElectionTask.stop();
        this.threadExecutor.shutdown();
        this.scheduledExecutorService.shutdown();
    }

    private List<String> getNodeIds(List<NodeHeartBeatData> allHeartbeatData) {
        ArrayList<String> allNodeIds = new ArrayList<String>(allHeartbeatData.size());
        for (NodeHeartBeatData nodeHeartBeatData : allHeartbeatData) {
            allNodeIds.add(nodeHeartBeatData.getNodeId());
        }
        return allNodeIds;
    }

    private class CoordinatorElectionTask
    implements Runnable {
        private boolean running = true;
        private ScheduledFuture<?> scheduledFuture;

        private CoordinatorElectionTask() {
        }

        @Override
        public void run() {
            while (this.running) {
                try {
                    if (RDBMSCoordinationStrategy.this.logger.isDebugEnabled()) {
                        RDBMSCoordinationStrategy.this.logger.debug((Object)("Current node state: " + (Object)((Object)RDBMSCoordinationStrategy.this.currentNodeState)));
                    }
                    switch (RDBMSCoordinationStrategy.this.currentNodeState) {
                        case CANDIDATE: {
                            RDBMSCoordinationStrategy.this.currentNodeState = this.performStandByTask();
                            break;
                        }
                        case COORDINATOR: {
                            RDBMSCoordinationStrategy.this.currentNodeState = this.performCoordinatorTask();
                            break;
                        }
                        case ELECTION: {
                            RDBMSCoordinationStrategy.this.currentNodeState = this.performElectionTask();
                        }
                    }
                }
                catch (Throwable e) {
                    RDBMSCoordinationStrategy.this.logger.error((Object)("Error detected while running coordination algorithm. Node became a " + (Object)((Object)NodeState.CANDIDATE) + " node"), e);
                    this.cancelStateExpirationTask();
                    RDBMSCoordinationStrategy.this.currentNodeState = NodeState.CANDIDATE;
                }
            }
        }

        private NodeState performStandByTask() throws AndesException, InterruptedException {
            NodeState nextState;
            this.updateNodeHeartBeat();
            boolean coordinatorValid = RDBMSCoordinationStrategy.this.contextStore.checkIfCoordinatorValid(RDBMSCoordinationStrategy.this.heartbeatMaxAge);
            TimeUnit.MILLISECONDS.sleep(RDBMSCoordinationStrategy.this.heartBeatInterval);
            if (coordinatorValid) {
                nextState = NodeState.CANDIDATE;
            } else {
                coordinatorValid = RDBMSCoordinationStrategy.this.contextStore.checkIfCoordinatorValid(RDBMSCoordinationStrategy.this.heartbeatMaxAge);
                if (coordinatorValid) {
                    nextState = NodeState.CANDIDATE;
                } else {
                    RDBMSCoordinationStrategy.this.logger.info((Object)"Going for election since the Coordinator is invalid");
                    RDBMSCoordinationStrategy.this.contextStore.removeCoordinator();
                    nextState = NodeState.ELECTION;
                }
            }
            return nextState;
        }

        private void updateNodeHeartBeat() throws AndesException {
            boolean heartbeatEntryExists = RDBMSCoordinationStrategy.this.contextStore.updateNodeHeartbeat(RDBMSCoordinationStrategy.this.localNodeId);
            if (!heartbeatEntryExists) {
                RDBMSCoordinationStrategy.this.contextStore.createNodeHeartbeatEntry(RDBMSCoordinationStrategy.this.localNodeId, RDBMSCoordinationStrategy.this.hazelcastAddress);
            }
        }

        private NodeState performCoordinatorTask() throws AndesException, InterruptedException {
            boolean stillCoordinator = RDBMSCoordinationStrategy.this.contextStore.updateCoordinatorHeartbeat(RDBMSCoordinationStrategy.this.localNodeId);
            if (stillCoordinator) {
                this.resetScheduleStateExpirationTask();
                long startTime = System.currentTimeMillis();
                this.updateNodeHeartBeat();
                long currentTimeMillis = System.currentTimeMillis();
                List<NodeHeartBeatData> allNodeInformation = RDBMSCoordinationStrategy.this.contextStore.getAllHeartBeatData();
                List allActiveNodeIds = RDBMSCoordinationStrategy.this.getNodeIds(allNodeInformation);
                ArrayList<String> newNodes = new ArrayList<String>();
                ArrayList<String> removedNodes = new ArrayList<String>();
                for (NodeHeartBeatData nodeHeartBeatData : allNodeInformation) {
                    long heartbeatAge = currentTimeMillis - nodeHeartBeatData.getLastHeartbeat();
                    String nodeId = nodeHeartBeatData.getNodeId();
                    if (nodeHeartBeatData.isNewNode()) {
                        newNodes.add(nodeId);
                        RDBMSCoordinationStrategy.this.contextStore.markNodeAsNotNew(nodeId);
                        continue;
                    }
                    if (heartbeatAge < (long)RDBMSCoordinationStrategy.this.heartbeatMaxAge) continue;
                    removedNodes.add(nodeId);
                    allActiveNodeIds.remove(nodeId);
                    RDBMSCoordinationStrategy.this.contextStore.removeNodeHeartbeat(nodeId);
                }
                for (String newNode : newNodes) {
                    RDBMSCoordinationStrategy.this.logger.info((Object)("Member added " + newNode));
                    RDBMSCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(allActiveNodeIds, MembershipEventType.MEMBER_ADDED, newNode);
                }
                for (String removedNode : removedNodes) {
                    RDBMSCoordinationStrategy.this.logger.info((Object)("Member removed " + removedNode));
                    RDBMSCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(allActiveNodeIds, MembershipEventType.MEMBER_REMOVED, removedNode);
                }
                long endTime = System.currentTimeMillis();
                long timeSpent = endTime - startTime;
                long timeToWait = (long)RDBMSCoordinationStrategy.this.heartBeatInterval - timeSpent;
                if (timeToWait > 0L) {
                    TimeUnit.MILLISECONDS.sleep(timeToWait);
                } else {
                    RDBMSCoordinationStrategy.this.logger.warn((Object)("Sending membership events took more than the heart beat interval. Time taken " + timeSpent + "ms"));
                }
                return NodeState.COORDINATOR;
            }
            RDBMSCoordinationStrategy.this.logger.info((Object)"Going for election since Coordinator state is lost");
            this.cancelStateExpirationTask();
            return NodeState.ELECTION;
        }

        private NodeState performElectionTask() throws InterruptedException {
            NodeState nextState;
            try {
                nextState = this.tryToElectSelfAsCoordinator();
            }
            catch (AndesException e) {
                RDBMSCoordinationStrategy.this.logger.info((Object)("Current node became a " + (Object)((Object)NodeState.CANDIDATE) + " node"), (Throwable)e);
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        private NodeState tryToElectSelfAsCoordinator() throws AndesException, InterruptedException {
            NodeState nextState;
            boolean electedAsCoordinator = RDBMSCoordinationStrategy.this.contextStore.createCoordinatorEntry(RDBMSCoordinationStrategy.this.localNodeId, RDBMSCoordinationStrategy.this.thriftAddress);
            if (electedAsCoordinator) {
                TimeUnit.MILLISECONDS.sleep(RDBMSCoordinationStrategy.this.coordinatorEntryCreationWaitTime);
                boolean isCoordinator = RDBMSCoordinationStrategy.this.contextStore.checkIsCoordinator(RDBMSCoordinationStrategy.this.localNodeId);
                if (isCoordinator) {
                    RDBMSCoordinationStrategy.this.contextStore.updateCoordinatorHeartbeat(RDBMSCoordinationStrategy.this.localNodeId);
                    this.resetScheduleStateExpirationTask();
                    RDBMSCoordinationStrategy.this.logger.info((Object)"Elected current node as the coordinator");
                    RDBMSCoordinationStrategy.this.configurableClusterAgent.becameCoordinator();
                    nextState = NodeState.COORDINATOR;
                    RDBMSCoordinationStrategy.this.membershipEventingEngine.notifyMembershipEvent(RDBMSCoordinationStrategy.this.getAllNodeIdentifiers(), MembershipEventType.COORDINATOR_CHANGED, RDBMSCoordinationStrategy.this.localNodeId);
                } else {
                    RDBMSCoordinationStrategy.this.logger.info((Object)("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node"));
                    nextState = NodeState.CANDIDATE;
                }
            } else {
                RDBMSCoordinationStrategy.this.logger.info((Object)("Election resulted in current node becoming a " + (Object)((Object)NodeState.CANDIDATE) + " node"));
                nextState = NodeState.CANDIDATE;
            }
            return nextState;
        }

        public void stop() {
            this.running = false;
        }

        private void cancelStateExpirationTask() {
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(true);
                this.scheduledFuture = null;
            }
        }

        private void resetScheduleStateExpirationTask() {
            this.cancelStateExpirationTask();
            this.scheduledFuture = RDBMSCoordinationStrategy.this.scheduledExecutorService.schedule(new Runnable(){

                @Override
                public void run() {
                    RDBMSCoordinationStrategy.this.currentNodeState = NodeState.ELECTION;
                }
            }, (long)RDBMSCoordinationStrategy.this.heartbeatMaxAge, TimeUnit.MILLISECONDS);
        }
    }

    private static enum NodeState {
        COORDINATOR,
        CANDIDATE,
        ELECTION;

    }
}

